<?php

namespace Move\Http\Strategy\Handler;

use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\Uri;
use Move\Http\Strategy\Serializer\JsonSerializer;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class RssHandlerServiceTest
 * @package Move\Http\Strategy\Handler
 */
class RssHandlerServiceTest extends TestCase
{

    protected  $tpl = 'rss/rss';


    /**
     * @return array
     */
    public function provideIsSatisfied()
    {
        return [
            ['application/xml', '/flux.xml', false],
            ['application/rss', '/flux.rss', false],
            ['application/rss+xml', '/flux-rss.xml', true],
            ['application/rss+xml', '/rss.xml', true],
        ];
    }

    /**
     * Create a new instance of the SUT
     * @return RssHandlerService
     */
    protected function createInstance()
    {
        // Should use mock...
        $serializer = new JsonSerializer();

        return new RssHandlerService($serializer, $this->tpl);
    }

    /**
     * @covers RssHandlerService::__construct
     * @return RssHandlerService
     */
    public function testConstruct()
    {
        $instance = $this->createInstance();
        $this->assertInstanceOf(RssHandlerService::class, $instance);
        return $instance;
    }

    /**
     * @dataProvider provideIsSatisfied
     * @covers RssHandlerService::isSatisfiedBy
     * @depends testConstruct
     *
     * @param $returnHeaderAccept
     * @param $returnUri
     * @param $expected
     *
     */
    public function testIsSatisfiedBy($returnHeaderAccept, $returnUri, $expected)
    {
        /** @var ServerRequestInterface |\PHPUnit_Framework_MockObject_MockObject $requeatInterface */
        $requeatInterface = $this->getMockBuilder(ServerRequestInterface::class)
            ->getMock();

        $requeatInterface->method('getUri')
            ->willReturn(new Uri($returnUri));

        $requeatInterface->method('getHeaderLine')
            ->with('Accept')
            ->will($this->returnValue($returnHeaderAccept));

        $service = $this->createInstance();
        $this->assertEquals($expected, $service->isSatisfiedBy($requeatInterface));
    }

    /**
     * @covers RssHandlerService::determineResponse
     * @depends testConstruct
     */
    public function testDetermineResponse()
    {
        $expectedContentType = 'application/rss+xml';

        // Should use mock...
        $responseInterface = new Response();
        $service = $this->createInstance();

        $alteredService = $service->determineResponse(
            ['data'=> ['head' => 'body']],
            $responseInterface
        );

        $this->assertInstanceOf(ResponseInterface::class, $alteredService);

        $contentType = $alteredService->getHeader('content-type');

        $this->assertArraySubset([$expectedContentType], $contentType);
    }
}
