<?php


namespace Cms\Mail;

use Move\Config;

/**
 * Class PhpMailerAdapter
 * @package Cms\Mail
 */
class PhpMailerAdapter implements SenderAdapterInterface
{

    /** @var \PHPMailer */
    private $mailer;

    /**
     * PhpMailerAdapter constructor.
     * @param Config $config
     * @throws \InvalidArgumentException
     * @throws \phpmailerException
     */
    public function __construct(Config $config)
    {
        // config
        if (!empty($config['smtp'])) {
            $this->mailer = self::createPhpMailerSmtp(
                $config['smtp.host'],
                !empty($config['smtp.port']) ? $config['smtp.port'] : null,
                !empty($config['smtp.security']) ? $config['smtp.security'] : false,
                !empty($config['smtp.username']) ? $config['smtp.username'] : null,
                !empty($config['smtp.password']) ? $config['smtp.password'] : null
            );
            $this->mailer->Debugoutput = 'error_log';
        } else {
            throw new \InvalidArgumentException('no configuration found for mailer');
        }

        // send from
        if (!empty($config['from.name']) && !empty($config['from.email'])) {
            $this->setFrom($config['from.name'], $config['from.email']);
        }

        // add CC
        foreach (@$config['copy_to'] ?? [] as $email) {
            $this->addCC($email);
        }
    }

    /**
     * @param string $name
     * @param string $email
     * @return $this
     * @throws \phpmailerException
     */
    public function setFrom(string $name, string $email)
    {
        $this->mailer->setFrom($email, $name);
        return $this;
    }

    /**
     * @param string $email
     * @param string|null $name
     * @return $this
     */
    public function addCC(string $email, string $name = null)
    {
        $this->mailer->addCC($email, $name);
        return $this;
    }

    /**
     * @param string $smtpHost
     * @param null|string $smtpPort
     * @param bool|string $security
     * @param null|string $authUser
     * @param null|string $authPasswd
     * @return \PHPMailer
     */
    public static function createPhpMailerSmtp(
        $smtpHost,
        $smtpPort = null,
        $security = false,
        $authUser = null,
        $authPasswd = null
    ) : \PHPMailer {
        $phpMailer = new \PHPMailer(true);
        $phpMailer->isSMTP();
        $phpMailer->Host = $smtpHost;
        if ($smtpPort) {
            $phpMailer->Port = $smtpPort;
        }
        if (!empty($security)) {
            $phpMailer->SMTPSecure = $security;
        }
        if ($authUser && $authPasswd) {
            $phpMailer->SMTPAuth = true;
            $phpMailer->Username = $authUser;
            $phpMailer->Password = $authPasswd;
        }
        return $phpMailer;
    }

    /**
     * @param array $to [name => address@mail]
     * @param string $subject
     * @param string $htmlBody
     * @param string|null $textBody
     * @param array $attachs
     * @return bool
     * @throws SendMailException
     * @throws \phpmailerException
     */
    public function sendHtmlMail(
        array $to,
        $subject,
        $htmlBody,
        $textBody = null,
        array $attachs = []
    ) {
        $mailer = clone $this->mailer;
        $mailer->CharSet = 'UTF-8';
        $mailer->Subject = $subject;

        $mailer->isHTML();
        $mailer->Body = $htmlBody;
        $mailer->AltBody = $textBody;

        foreach ($to as $name => $address) {
            $mailer->addAddress($address, $name);
        }

        foreach ($attachs as $name => $filepath) {
            if (!is_file($filepath)) {
                throw new \UnexpectedValueException("$filepath is not a file");
            }
            $mailer->addAttachment($filepath, $name);
        }

        try {
            return $mailer->send();
        } catch (\phpmailerException $e) {
            throw new SendMailException((string)$e, $e->getCode(), $e);
        }
    }
}
