<?php

namespace Cms\DashParser\Serializer;

use Cms\DashParser\Block\BlockAtomic;
use Cms\DashParser\Block\BlockBlockquote;
use Cms\DashParser\Block\BlockBreakline;
use Cms\DashParser\Block\BlockHeadline;
use Cms\DashParser\Block\BlockInterface;
use Cms\DashParser\Block\BlockListItem;
use Cms\DashParser\Block\BlockParagraph;

/**
 * Class HtmlBlockConverter
 * @package Cms\DashParser\Serializer
 */
class HtmlBlockConverter
{
    /** @var bool|BlockListItem */
    private $listOpen = false;

    /**
     * @param BlockInterface $block
     * @param string $html
     * @return null|string
     */
    public function createBlock(BlockInterface $block, &$html)
    {
        // bloc atomic on laisse l'entité faire ce qu'elle veux
        if ($block instanceof BlockAtomic) {
            return '';
        }

        // saut de ligne on stop direct apres
        if ($block instanceof BlockBreakline) {
            $html .= $this->createBreakLine();
            return null;
        }

        // bloc liste
        if ($block instanceof BlockListItem) {
            $html .= $this->beginListItemBlock($block);
            $blockHtml = $this->createListItem($block);
        } // type de bloc simple
        else {
            // fermeture de la list si existante
            $html .= $this->endListItemBlock();
            // gestion des blocks
            if ($block instanceof BlockHeadline) {
                $blockHtml = $this->createHeader($block);
            } elseif ($block instanceof BlockBlockquote) {
                $blockHtml = $this->createBlockquote($block);
            } elseif ($block instanceof BlockParagraph) {
                $blockHtml = $this->createParagraph($block);
            }
        }
        if (isset($blockHtml)) {
            return $blockHtml;
        }
        return null;
    }

    /**
     * @return bool|BlockListItem
     */
    public function getListOpen()
    {
        return $this->listOpen;
    }

    /**
     * @param bool|BlockListItem $listOpen
     */
    public function setListOpen($listOpen)
    {
        $this->listOpen = $listOpen;
    }

    /**
     * @param BlockListItem $block
     * @return string
     */
    protected function beginListItemBlock(BlockListItem $block)
    {
        if ($this->getListOpen() === false) {
            $this->setListOpen($block);
            return $block->isOrdered() ? "<ol>" : "<ul>";
        }
        return '';
    }

    /**
     * @return bool|string
     */
    protected function endListItemBlock()
    {
        // fermeture de la list si existante
        if ($this->getListOpen() instanceof BlockListItem) {
            $html = $this->getListOpen()->isOrdered() ? "</ol>" : "</ul>";
            $this->setListOpen(false);
            return $html;
        }
        return '';
    }

    /**
     * @return string
     */
    protected function createBreakLine()
    {
        return "<p>&nbsp;</p>";
    }

    /**
     * @param BlockListItem $block
     * @return string
     */
    protected function createListItem(BlockListItem $block)
    {
        $contentText = $block->getText();
        return sprintf("<li>%s</li>", $contentText);
    }

    /**
     * @param BlockParagraph $block
     * @return string
     */
    protected function createParagraph(BlockParagraph $block)
    {
        $contentText = $block->getText();
        return sprintf("<p>%s</p>", $contentText);
    }

    /**
     * @param BlockHeadline $block
     * @return string
     */
    public function createHeader(BlockHeadline $block)
    {
        $contentText = $block->getHeadLine();
        $level = $block->getLevel();

        $html = <<<EOF
<h$level>%s</h$level>
EOF;
        return sprintf($html, $contentText);
    }

    /**
     * @param BlockBlockquote $block
     * @return string
     */
    public function createBlockquote(BlockBlockquote $block)
    {
        $contentText = $block->getText();
        return sprintf("<blockquote>%s</blockquote>", $contentText);
    }
}
