<?php


namespace Cms\DashParser\Serializer;

use Cms\DashParser\Entity\EntityEmbed;
use Cms\DashParser\Entity\EntityImage;
use Cms\DashParser\Entity\EntityInterface;
use Cms\DashParser\Entity\EntityLinkableInterface;
use Cms\DashParser\Entity\EntityPositionableInterface;
use Cms\DashParser\Entity\EntityResource;

/**
 * Class HtmlEntityConverter
 * @package Cms\DashParser\Serializer
 */
class HtmlEntityConverter implements HtmlEntityConverterInterface
{

    /** @var HtmlEntityImageConverter */
    protected $imageConverter;

    /** @var HtmlEntityEmbedConverter */
    protected $embedConverter;

    /** @var HtmlEntityLinkConverter */
    protected $linkConverter;

    /**
     * HtmlEntityConverter constructor.
     * @param HtmlEntityImageConverter $imageConverter
     * @param HtmlEntityEmbedConverter $embedConverter
     * @param HtmlEntityLinkConverter $linkConverter
     */
    public function __construct(
        HtmlEntityImageConverter $imageConverter = null,
        HtmlEntityEmbedConverter $embedConverter = null,
        HtmlEntityLinkConverter $linkConverter = null
    ) {
        $this->imageConverter = $imageConverter ?: new HtmlEntityImageConverter();
        $this->embedConverter = $embedConverter ?: new HtmlEntityEmbedConverter();
        $this->linkConverter = $linkConverter ?: new HtmlEntityLinkConverter();
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityLinkableInterface $entity
     * @param string $nodeText
     * @return \DOMElement|null
     */
    public function createLinkFromEntity(\DOMDocument $doc, EntityLinkableInterface $entity, $nodeText)
    {
        return $this->linkConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityImage $entity
     * @param string $nodeText
     * @return \DOMElement|null
     */
    public function createImageFromEntity(\DOMDocument $doc, EntityImage $entity, $nodeText)
    {
        return $this->imageConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityEmbed $entity
     * @param string $nodeText
     * @return \DOMElement
     */
    public function createEmbedFromEntity(\DOMDocument $doc, EntityEmbed $entity, $nodeText)
    {
        return $this->embedConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityResource $entity
     * @param string $nodeText
     * @return \DOMElement
     */
    public function createResourceFromEntity(\DOMDocument $doc, EntityResource $entity, $nodeText)
    {
        $block = $doc->createElement('p', $entity->getLabel());
        $block->setAttribute('style', 'display: none');
        $block->setAttribute('data-ref', $entity->getRef());
        $block->setAttribute('data-type', $entity->getType());
        $block->setAttribute('data-provider', $entity->getProvider());
        return $block;
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityInterface $entity
     * @param string $nodeText
     * @return \DOMElement
     */
    public function getEntityNode(\DOMDocument $doc, EntityInterface $entity, $nodeText = null)
    {
        $classValue = [];
        if ($entity instanceof EntityPositionableInterface) {
            $classValue[] = 'align-' . $entity->getPosition();
        }

        // gstion des noeud resource
        if ($entity instanceof EntityResource) {
            $node = $this->createResourceFromEntity($doc, $entity, $nodeText);
        } // gstion des noeud embed
        elseif ($entity instanceof EntityEmbed) {
            $node = $this->createEmbedFromEntity($doc, $entity, $nodeText);
        } // gestion des noeud images
        elseif ($entity instanceof EntityImage) {
            $imgNode = $this->createImageFromEntity($doc, $entity, $nodeText);
            $linkNode = $this->createLinkFromEntity($doc, $entity, $nodeText);
            if ($linkNode !== null) {
                $linkNode->appendChild($imgNode);
                $node = $linkNode;
            } else {
                $node = $imgNode;
            }
        } // gestion des noeud liens
        elseif ($entity instanceof EntityLinkableInterface) {
            $node = $this->createLinkFromEntity($doc, $entity, $nodeText);
        }

        // noeud par default
        if (!isset($node)) {
            $node = $doc->createElement('span');
        }

        // ajout des attributs commun
        if (!empty($classValue)) {
            $node->setAttribute("class", $node->getAttribute('class') . ' ' . implode(' ', $classValue));
        }
        return $node;
    }
}
