<?php

namespace Cms\DashParser\Serializer;

use Cms\DashParser\Block\BlockInterface;
use Cms\DashParser\Block\BlockTextAwareInterface;
use Cms\DashParser\Entity\EntityBlockInterface;
use Cms\DashParser\Entity\EntityInterface;

/**
 * Class HtmlSerializer
 * @package Cms\DashParser\Serializer
 */
class HtmlSerializer implements SerializerInterface
{

    /** @var  EntityInterface[] */
    private $entityCollection;

    /** @var HtmlBlockConverter */
    private $blockConverter;

    /** @var HtmlEntityConverter */
    private $entityConverter;

    /** @var HtmlStyleConverter */
    private $styleConverter;

    /**
     * HtmlSerializer constructor.
     * @param array $entityCollection
     * @param HtmlEntityConverter $entitySerializer
     * @param HtmlBlockConverter $blockSerializer
     * @param HtmlStyleConverter $styleSerializer
     */
    public function __construct(
        array $entityCollection,
        HtmlEntityConverter $entitySerializer = null,
        HtmlBlockConverter $blockSerializer = null,
        HtmlStyleConverter $styleSerializer = null
    ) {
        $this->entityCollection = $entityCollection;
        $this->entityConverter = $entitySerializer ?: new HtmlEntityConverter();
        $this->blockConverter = $blockSerializer ?: new HtmlBlockConverter();
        $this->styleConverter = $styleSerializer ?: new HtmlStyleConverter();
    }


    /**
     * @param string $key
     * @return EntityInterface|null
     */
    public function getEntityByKey($key)
    {
        // check exist entity
        if (!isset($this->entityCollection[$key])) {
            return null;
        }
        return $this->entityCollection[$key];
    }

    /**
     * @param string $text
     * @param array $chunkPos
     * @param array $chunk
     * @return array
     */
    protected function chunkTextInto($text, array $chunkPos, array $chunk = [])
    {
        // découpe du texte
        $chunkPos = array_unique($chunkPos);
        sort($chunkPos);
        foreach ($chunkPos as $k => $offset) {
            if (!isset($chunkPos[$k + 1])) {
                break;
            }
            $pos = $chunkPos[$k + 1] - $offset;
            $chunk[$offset][] = mb_substr($text, 0, $pos);
            $text = mb_substr($text, $pos);
        }

        ksort($chunk);
        return $chunk;
    }

    /**
     * @param string $nodeName
     * @param \Iterator|array $attrs
     * @param int $offsetBegin
     * @param int $length
     * @param array $chunk
     * @return array
     */
    protected function chunkForNodeInto(
        $nodeName,
        $attrs,
        $offsetBegin,
        $length,
        array $chunk = []
    ) {
        // construction des attribut de la balise
        $attrsHtml = '';
        foreach ($attrs as $name => $attr) {
            $attrsHtml .= ' ' . $name . '="' . str_replace('"', '&quote;', $attr) . '"';
        }

        // ajout de la balise d'ouverture
        if (!isset($chunk[$offsetBegin])) {
            $chunk[$offsetBegin] = [];
        }
        if (!isset($chunk[$offsetBegin][$length])) {
            $chunk[$offsetBegin][$length] = [];
        }
        $chunk[$offsetBegin][$length][] = "<$nodeName$attrsHtml>";

        // trie de la section pour eviter les melange d'open/close de balise
        usort($chunk[$offsetBegin][$length], function ($prev, $nex) {
            preg_match("@<([a-z0-9]+)@i", $prev, $matchPrev);
            preg_match("@<([a-z0-9]+)@i", $nex, $matchNex);
            return strcmp($matchPrev[1], $matchNex[1]);
        });
        krsort($chunk[$offsetBegin]);

        // ajout de la balise de fermeture
        $offsetEnd = $offsetBegin + $length;
        if (!isset($chunk[$offsetEnd])) {
            $chunk[$offsetEnd] = [];
        }
        if (!isset($chunk[$offsetEnd][$length])) {
            $chunk[$offsetEnd][$length] = [];
        }
        $chunk[$offsetEnd][$length][] = "</$nodeName>";

        // trie de la section pour eviter les melange d'open/close de balise
        usort($chunk[$offsetEnd][$length], function ($prev, $nex) {
            preg_match("@</([a-z0-9]+)@i", $prev, $matchPrev);
            preg_match("@</([a-z0-9]+)@i", $nex, $matchNex);
            return strcmp($matchPrev[1], $matchNex[1]);
        });
        rsort($chunk[$offsetEnd][$length]);
        ksort($chunk[$offsetEnd]);

        return $chunk;
    }

    /**
     * @param string $contentHtml
     * @param array $styleRanges
     * @param array $chunk
     * @return array
     */
    protected function chunkWithStyle($contentHtml, array $styleRanges, array $chunk = [])
    {
        foreach ($styleRanges as $inlineStyle) {
            // detection de la balise
            $nodeName = $this->styleConverter->getStyleNodeName($inlineStyle['style'], $contentHtml);
            if ($nodeName === null) {
                continue;
            }

            // calcul des offset
            $offsetBegin = $inlineStyle['offset'];

            // ajoute un chunk de balise
            $chunk = $this->chunkForNodeInto(
                $nodeName,
                [],
                $offsetBegin,
                $inlineStyle['length'],
                $chunk
            );
        }

        return $chunk;
    }

    /**
     * @param string $contentHtml
     * @param array $entityRanges
     * @param array $chunk
     * @return array
     */
    protected function chunkWithEntity($contentHtml, array $entityRanges, array $chunk = [])
    {
        $doc = $this->createDomDocument();
        foreach ($entityRanges as $key => $entityRange) {
            // check exist entity
            $entity = $this->getEntityByKey($entityRange['key']);
            if ($entity === null) {
                continue;
            }

            // calcul des offset
            $offsetBegin = $entityRange['offset'];
            $offsetEnd = $offsetBegin + $entityRange['length'];
            $contentHtml = mb_substr($contentHtml, $offsetBegin, $entityRange['length']);

            // detection de la balise
            $element = $this->entityConverter->getEntityNode($doc, $entity, $contentHtml);
            if ($element === null) {
                continue;
            }

            // is blockEntity
            if ($entity instanceof EntityBlockInterface) {
                if (!isset($chunk[$offsetEnd])) {
                    $chunk[$offsetEnd] = [];
                }
                $chunk[$offsetEnd][] = $doc->saveXML($element);
                continue;
            }

            // ajoute un chunk de balise
            $attrs = [];
            foreach ($element->attributes as $attr) {
                $attrs[$attr->nodeName] = $attr->nodeValue;
            }
            $chunk = $this->chunkForNodeInto(
                $element->nodeName,
                $attrs,
                $offsetBegin,
                $entityRange['length'],
                $chunk
            );
        }
        unset($doc);

        return $chunk;
    }

    /**
     * @param BlockInterface[] $blockCollection
     * @return string
     */
    public function serialize(array $blockCollection = [])
    {
        $html = '';
        foreach ($blockCollection as $block) {
            // check type
            if (!$block instanceof BlockInterface) {
                continue;
            }

            // creation du block
            $blockHtml = $this->blockConverter->createBlock($block, $html);
            if ($blockHtml === null) {
                continue;
            }

            // decoupe du block texte
            $contentHtml = '';
            $doc = $this->createDomDocument();
            if ($block instanceof BlockTextAwareInterface) {
                $contentHtml = $block->getText();
                // patch pour les anciens bloc
                if ($contentHtml === 'image' || $contentHtml === 'embed') {
                    $contentHtml = '';
                } elseif (!empty($blockHtml)) {
                    // fix pour les & seul
                    self::fixAmps($blockHtml);
                    // convertion du contenu en UTF-8
                    $blockHtml = mb_convert_encoding($blockHtml, 'HTML-ENTITIES', 'UTF-8');
                    $xmlOpt = LIBXML_COMPACT | LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD;
                    $doc->loadHTML($blockHtml, $xmlOpt);
                }
            }

            // chuck pour les styles
            $chunk = $this->chunkWithStyle($contentHtml, $block->getInlineStyleRanges());

            // chunk pour les entities
            $chunk = $this->chunkWithEntity($contentHtml, $block->getEntityRanges(), $chunk);

            // découpe du texte
            if (count($chunk) === 0) {
                $chunk[0] = $contentHtml;
            } else {
                // ajoute un point sur le premier char
                if (!isset($chunk[0])) {
                    $chunk[0] = [];
                }
                // ajoute un point sur le dernier char
                $contentLength = mb_strlen($contentHtml);
                if (!isset($chunk[$contentLength])) {
                    $chunk[$contentLength] = [];
                }
                // découpe le texte en suivant les chunk
                $chunk = $this->chunkTextInto($contentHtml, array_keys($chunk), $chunk);
            }

            // flatten array
            ksort($chunk);
            $chunk = new \RecursiveIteratorIterator(new \RecursiveArrayIterator($chunk));
            $chunk = iterator_to_array($chunk, false);

            // mise a jour du block
            $contentHtml = nl2br(implode('', $chunk));
            if (!empty($contentHtml)) {
                $frag = $doc->createDocumentFragment();
                // fix pour les & seul
                self::fixAmps($contentHtml);
                if (!$frag->appendXML($contentHtml)) {
                    trigger_error("Invalid fragment : " . $contentHtml, E_USER_WARNING);
                } else {
                    if ($doc->firstChild) {
                        $doc->firstChild->textContent = null;
                        $doc->firstChild->appendChild($frag);
                    } else {
                        $doc->appendChild($frag);
                    }
                }
            }

            // ajoute le block
            $html .= $doc->saveHTML($doc->firstChild);
        }
        return $html;
    }

    /**
     * @param string $html
     * @param int $offset
     */
    protected static function fixAmps(&$html, $offset = 0)
    {
        $positionAmp = strpos($html, '&', $offset);
        if ($positionAmp !== false) { // If an '&' can be found.
            $positionSemiColumn = strpos($html, ';', $positionAmp + 1);
            $string = substr($html, $positionAmp, $positionSemiColumn - $positionAmp + 1);
            // If a standard escape cannot be found.
            if ($positionSemiColumn === false
                || preg_match('/&(#[0-9]+|[A-Z|a-z|0-9]+);/', $string) === 0
            ) {
                // This mean we need to escapa the '&' sign.
                $html = substr_replace($html, '&amp;', $positionAmp, 1);
                // Recursive call from the new position.
                self::fixAmps($html, $positionAmp + 5);
            } else {
                // Recursive call from the new position.
                self::fixAmps($html, $positionAmp + 1);
            }
        }
    }

    /**
     * @return \DOMDocument
     */
    protected function createDomDocument()
    {
        $imp = new \DOMImplementation();
        $doc = $imp->createDocument();
        // config xml
        $doc->encoding = 'UTF-8';
        // don't want to bother with white spaces
        $doc->preserveWhiteSpace = false;
        // invalid markup...
        $doc->strictErrorChecking = false;
        $doc->recover = true;
        return $doc;
    }

    /**
     * @param \DOMNode $element
     * @return string
     */
    protected function DOMinnerHTML(\DOMNode $element)
    {
        $innerHTML = "";
        $children = $element->childNodes;
        foreach ($children as $child) {
            $innerHTML .= $element->ownerDocument->saveHTML($child);
        }
        return $innerHTML;
    }
}
