<?php

namespace Cms\DashParser\Serializer;

use Cms\DashParser\Block\BlockInterface;
use Cms\DashParser\Block\BlockTextAwareInterface;
use Cms\DashParser\Entity\EntityBlockInterface;
use Cms\DashParser\Entity\EntityInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;
use function Cms\DashParser\load_html;
use function Cms\DashParser\fix_amps;
use function Cms\DashParser\html_specialchars;

/**
 * Class HtmlSerializer
 * @package Cms\DashParser\Serializer
 */
class HtmlSerializer implements SerializerInterface, LoggerAwareInterface
{

    use LoggerAwareTrait;

    /** @var  EntityInterface[] */
    private $entityCollection;

    /** @var HtmlBlockConverter */
    private $blockConverter;

    /** @var HtmlEntityConverter */
    private $entityConverter;

    /** @var HtmlStyleConverter */
    private $styleConverter;

    /**
     * HtmlSerializer constructor.
     * @param array $entityCollection
     * @param HtmlEntityConverter $entitySerializer
     * @param HtmlBlockConverter $blockSerializer
     * @param HtmlStyleConverter $styleSerializer
     */
    public function __construct(
        array $entityCollection,
        HtmlEntityConverter $entitySerializer = null,
        HtmlBlockConverter $blockSerializer = null,
        HtmlStyleConverter $styleSerializer = null
    ) {
        $this->entityCollection = $entityCollection;
        $this->entityConverter = $entitySerializer ?: new HtmlEntityConverter();
        $this->blockConverter = $blockSerializer ?: new HtmlBlockConverter();
        $this->styleConverter = $styleSerializer ?: new HtmlStyleConverter();
        $this->logger = new NullLogger();
    }


    /**
     * @param string $key
     * @return EntityInterface|null
     */
    public function getEntityByKey($key)
    {
        // check exist entity
        if (!isset($this->entityCollection[$key])) {
            return null;
        }
        return $this->entityCollection[$key];
    }

    /**
     * @param string $text
     * @param array $chunkPos
     * @param array $chunk
     * @return array
     */
    protected function chunkTextInto($text, array $chunkPos, array $chunk = [])
    {
        // découpe du texte
        $chunkPos = array_unique($chunkPos);
        sort($chunkPos);
        foreach ($chunkPos as $k => $offset) {
            if (!isset($chunkPos[$k + 1])) {
                break;
            }
            $pos = $chunkPos[$k + 1] - $offset;

            $textChunk = mb_substr($text, 0, $pos);
            //html_entities($textChunk);
            $chunk[$offset][] = $textChunk;

            $text = mb_substr($text, $pos);
        }

        ksort($chunk);
        return $chunk;
    }

    /**
     * @param string $nodeName
     * @param \Iterator|array $attrs
     * @param int $offsetBegin
     * @param int $length
     * @param array $chunk
     * @return array
     */
    protected function chunkForNodeInto(
        $nodeName,
        $attrs,
        $offsetBegin,
        $length,
        array $chunk = []
    ) {
        // construction des attribut de la balise
        $attrsHtml = '';
        foreach ($attrs as $name => $attr) {
            if (!empty($attr)) {
                $attrsHtml .= ' ' . $name;
                if (!is_bool($attr)) {
                    $attrsHtml .= '="' . str_replace('"', '&quote;', $attr) . '"';
                }
            }
        }

        // ajout de la balise d'ouverture
        $offsetEnd = $offsetBegin + $length;
        if (!isset($chunk[$offsetBegin])) {
            $chunk[$offsetBegin] = [];
        }
        if (!isset($chunk[$offsetBegin][$length])) {
            $chunk[$offsetBegin][$length] = [];
        }
        $this->logger->debug("open new node : $nodeName with attributes ($attrsHtml)", [
            'offsetBegin' => $offsetBegin,
            'offsetEnd' => $offsetEnd,
        ]);
        $chunk[$offsetBegin][$length][] = "<$nodeName$attrsHtml>";

        // si balise ouvrante/fermante on stop
        if ((int)$length === 0) {
            return $chunk;
        }

        // check si des balise se ferme avant moi pour evité les collisions
        ksort($chunk);
        $offsetEndInserted = false;
        foreach ($chunk as $offset => &$item) {
            // on demarre a mon offset de depart
            if ($offset <= $offsetBegin) {
                continue;
            }
            // on s'arrete a mon offset de fin
            if ($offset > $offsetEnd) {
                break;
            }
            if (!is_array($item)) {
                continue;
            }
            foreach ($item as &$node) {
                $this->logger->debug("close node $nodeName before other node open : " . $node[0], [
                    'offsetBegin' => $offsetBegin,
                    'offsetEnd' => $offsetEnd,
                ]);
                array_unshift($node, "</$nodeName>");
                // on ajoute pas la dernière balise sinon on devrai la refermé après
                if ($offset < $offsetEnd) {
                    $this->logger->debug("open node after closed : $nodeName with attributes ($attrsHtml)", [
                        'offsetBegin' => $offsetBegin,
                        'offsetEnd' => $offsetEnd,
                    ]);
                    $node[] = "<$nodeName$attrsHtml>";
                } else {
                    $offsetEndInserted = true;
                }
            }
            unset($node);
        }
        unset($item);

        // ajout de la balise de fermeture
        if ($offsetEndInserted === false) {
            if (!isset($chunk[$offsetEnd])) {
                $chunk[$offsetEnd] = [];
            }
            if (!isset($chunk[$offsetEnd][$length])) {
                $chunk[$offsetEnd][$length] = [];
            }
            $this->logger->debug("close node : $nodeName", [
                'offsetBegin' => $offsetBegin,
                'offsetEnd' => $offsetEnd,
            ]);
            $chunk[$offsetEnd][$length][] = "</$nodeName>";
        }
        return $chunk;
    }

    /**
     * @param string $contentHtml
     * @param array $styleRanges
     * @param array $chunk
     * @return array
     */
    protected function chunkWithStyle($contentHtml, array $styleRanges, array $chunk = [])
    {
        foreach ($styleRanges as $inlineStyle) {
            // detection de la balise
            $nodeName = $this->styleConverter->getStyleNodeName(
                $inlineStyle['style'],
                $contentHtml
            );
            if ($nodeName === null) {
                continue;
            }

            // calcul des offset
            $offsetBegin = $inlineStyle['offset'];
            $this->logger->debug("new style detected : $nodeName", [
                'contentHtml' => $contentHtml,
                'offsetBegin' => $offsetBegin,
            ]);

            // ajoute un chunk de balise
            $chunk = $this->chunkForNodeInto(
                $nodeName,
                [],
                $offsetBegin,
                $inlineStyle['length'],
                $chunk
            );
        }

        return $chunk;
    }

    /**
     * @param string $contentHtml
     * @param array $entityRanges
     * @param array $chunk
     * @return array
     */
    protected function chunkWithEntity($contentHtml, array $entityRanges, array $chunk = [])
    {
        $doc = $this->createDomDocument();
        foreach ($entityRanges as $key => $entityRange) {
            // check exist entity
            $entity = $this->getEntityByKey($entityRange['key']);
            if ($entity === null) {
                continue;
            }

            // calcul des offset
            $offsetBegin = $entityRange['offset'];
            $offsetEnd = $offsetBegin + $entityRange['length'];
            $contentHtml = mb_substr($contentHtml, $offsetBegin, $entityRange['length']);
            $this->logger->debug('new entity detected : ' . get_class($entity), [
                'contentHtml' => $contentHtml,
                'offsetBegin' => $offsetBegin,
                'offsetEnd' => $offsetEnd,
            ]);

            // detection de la balise
            $element = $this->entityConverter->getEntityNode($doc, $entity, $contentHtml);
            if ($element === null) {
                continue;
            }

            // is blockEntity
            if ($entity instanceof EntityBlockInterface) {
                if (!isset($chunk[$offsetEnd])) {
                    $chunk[$offsetEnd] = [];
                }
                $chunk[$offsetEnd][] = $doc->saveXML($element);
                continue;
            }

            // ajoute un chunk de balise
            $attrs = [];
            foreach ($element->attributes as $attr) {
                $attrs[$attr->nodeName] = $attr->nodeValue;
            }
            $chunk = $this->chunkForNodeInto(
                $element->nodeName,
                $attrs,
                $offsetBegin,
                $entityRange['length'],
                $chunk
            );
        }
        unset($doc);

        return $chunk;
    }

    /**
     * @param BlockInterface[] $blockCollection
     * @return string
     */
    public function serialize(array $blockCollection = [])
    {
        $html = '';
        foreach ($blockCollection as $block) {
            // check type
            if (!$block instanceof BlockInterface) {
                continue;
            }

            // creation du block
            $blockHtml = $this->blockConverter->createBlock($block, $html);
            // aucun bloc découvert on passe au suivant (ou saut de ligne)
            if ($blockHtml === null) {
                continue;
            }
            $this->logger->debug('new block detected : ' . get_class($block), [
                'blockHtml' => $blockHtml,
            ]);

            // decoupe du block texte
            $contentHtml = '';
            $doc = $this->createDomDocument();
            if ($block instanceof BlockTextAwareInterface) {
                $contentHtml = $block->getText();
                // patch pour les anciens bloc
                if ($contentHtml === 'image' || $contentHtml === 'embed') {
                    $contentHtml = '';
                } elseif ('' !== $blockHtml) {
                    load_html($doc, $blockHtml);
                }
            }

            // chuck pour les styles
            $chunk = $this->chunkWithStyle($contentHtml, $block->getInlineStyleRanges());

            // chunk pour les entities
            $chunk = $this->chunkWithEntity($contentHtml, $block->getEntityRanges(), $chunk);

            // découpe du texte
            if (count($chunk) === 0) {
                html_specialchars($contentHtml);
                $chunk[0] = $contentHtml;
            } else {
                // ajoute un point sur le premier char
                if (!isset($chunk[0])) {
                    $chunk[0] = [];
                }
                // ajoute un point sur le dernier char
                $contentLength = mb_strlen($contentHtml);
                if (!isset($chunk[$contentLength])) {
                    $chunk[$contentLength] = [];
                }
                // découpe le texte en suivant les chunk
                $chunk = $this->chunkTextInto($contentHtml, array_keys($chunk), $chunk);
            }

            // flatten array
            ksort($chunk);
            $chunk = new \RecursiveIteratorIterator(new \RecursiveArrayIterator($chunk));
            $chunk = iterator_to_array($chunk, false);
            $this->logger->debug('chunk for block is finalized : ' . get_class($block), [
                'chunk' => $chunk,
            ]);

            // mise a jour du block
            $contentHtml = nl2br(implode('', $chunk));
            if (!empty($contentHtml)) {
                $body = $this->createDomDocument();
                if (!load_html($body, $contentHtml)) {
                    trigger_error('Invalid fragment : ' . $contentHtml, E_USER_WARNING);
                } else {
                    $doc = $body;
                }
            }

            // ajoute le block
            $html .= $doc->saveHTML($doc->firstChild);
        }
        return $html;
    }

    /**
     * @param string $html
     * @param int $offset
     * @deprecated
     */
    protected static function fixAmps(&$html, $offset = 0)
    {
        fix_amps($html, $offset);
    }

    /**
     * @return \DOMDocument
     */
    protected function createDomDocument()
    {
        $imp = new \DOMImplementation();
        $doc = $imp->createDocument();
        // config xml
        $doc->encoding = 'UTF-8';
        // don't want to bother with white spaces
        $doc->preserveWhiteSpace = false;
        // invalid markup...
        $doc->strictErrorChecking = false;
        $doc->recover = true;
        return $doc;
    }

    /**
     * @param \DOMNode $element
     * @return string
     */
    protected function DOMinnerHTML(\DOMNode $element) : string
    {
        $innerHTML = '';
        $children = $element->childNodes;
        foreach ($children as $child) {
            $innerHTML .= $element->ownerDocument->saveHTML($child);
        }
        return $innerHTML;
    }
}
