<?php

namespace Cms\DashParser;

/**
 * @param string $html
 * @param int $offset
 */
function fix_amps(&$html, $offset = 0)
{
    $positionAmp = strpos($html, '&', $offset);
    if ($positionAmp !== false) { // If an '&' can be found.
        $positionSemiColumn = strpos($html, ';', $positionAmp + 1);
        $string = substr($html, $positionAmp, $positionSemiColumn - $positionAmp + 1);
        // If a standard escape cannot be found.
        if ($positionSemiColumn === false
            || preg_match('/&(#\d+|[A-Z|a-z|0-9]+);/', $string) === 0
        ) {
            // This mean we need to escapa the '&' sign.
            $html = substr_replace($html, '&amp;', $positionAmp, 1);
            // Recursive call from the new position.
            fix_amps($html, $positionAmp + 5);
        } else {
            // Recursive call from the new position.
            fix_amps($html, $positionAmp + 1);
        }
    }
}

/**
 * @param string $html
 */
function html_specialchars(&$html)
{
    $opt = ENT_NOQUOTES | ENT_HTML5 | ENT_SUBSTITUTE;
    $html = \htmlspecialchars($html, $opt, 'UTF-8');
    fix_amps($html);
}

/**
 * @return \DOMDocument
 */
function create_document() : \DOMDocument
{
    $imp = new \DOMImplementation();
    $doc = $imp->createDocument();
    // config xml
    $doc->encoding = 'UTF-8';
    // don't want to bother with white spaces
    $doc->preserveWhiteSpace = false;
    // invalid markup...
    $doc->strictErrorChecking = false;
    $doc->recover = true;
    return $doc;
}

/**
 * @param \DOMDocument $doc
 * @param string $htmlContent
 * @param bool $hideError
 * @return bool
 */
function load_html(\DOMDocument $doc, string $htmlContent, bool $hideError = true) : bool
{
    $htmlContent = trim($htmlContent);
    if (empty($htmlContent)) {
        return true;
    }
    // fix pour les & seul
    fix_amps($htmlContent);
    // convertion
    $xmlOpt = LIBXML_COMPACT | LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD;
    $htmlContent = mb_convert_encoding($htmlContent, 'HTML-ENTITIES', 'UTF-8');
    // évite les erreurs de balises HTML5 inconnu
    libxml_use_internal_errors($hideError);
    $result = $doc->loadHTML($htmlContent, $xmlOpt);
    // recupere les erreur de load
    $errors = array_filter(libxml_get_errors(), static function ($err) {
        return !preg_match('@^Tag \w+ invalid@i', $err->message);
    });
    libxml_clear_errors();
    foreach ($errors as $xmlError) {
        trigger_error(
            'LoadHtml Error : ' . $xmlError->message . ' with content : ' . $htmlContent,
            E_USER_NOTICE
        );
    }
    // retabli les erreurs standards
    libxml_use_internal_errors(false);
    return $result;
}
