<?php


namespace Cms\DashParser\Serializer;

use Cms\DashParser\Entity\EntityDocument;
use Cms\DashParser\Entity\EntityEmbed;
use Cms\DashParser\Entity\EntityImage;
use Cms\DashParser\Entity\EntityInterface;
use Cms\DashParser\Entity\EntityLinkableInterface;
use Cms\DashParser\Entity\EntityPositionableInterface;
use Cms\DashParser\Entity\EntityResource;
use Cms\DashParser\Entity\EntityTable;
use Cms\DashParser\Entity\EntityTopList;

/**
 * Class HtmlEntityConverter
 * @package Cms\DashParser\Serializer
 */
class HtmlEntityConverter implements HtmlEntityConverterInterface
{

    /** @var HtmlEntityImageConverter */
    protected $imageConverter;

    /** @var HtmlEntityEmbedConverter */
    protected $embedConverter;

    /** @var HtmlEntityLinkConverter */
    protected $linkConverter;

    /** @var HtmlEntityResourceConverter */
    protected $resourceConverter;

    /** @var HtmlEntityDocumentConverter  */
    private $documentConverter;

    /** @var HtmlEntityTableConverter */
    private $tableConverter;

    /** @var HtmlEntityTopListConverter */
    protected $topListConverter;

    /**
     * HtmlEntityConverter constructor.
     * @param HtmlEntityImageConverter|null $imageConverter
     * @param HtmlEntityEmbedConverter|null $embedConverter
     * @param HtmlEntityLinkConverter|null $linkConverter
     * @param HtmlEntityResourceConverter|null $resourceConverter
     * @param HtmlEntityDocumentConverter|null $documentConverter
     * @param HtmlEntityTableConverter|null $tableConverter
     * @param HtmlEntityTopListConverter|null $topListConverter
     */
    public function __construct(
        HtmlEntityImageConverter $imageConverter = null,
        HtmlEntityEmbedConverter $embedConverter = null,
        HtmlEntityLinkConverter $linkConverter = null,
        HtmlEntityResourceConverter $resourceConverter = null,
        HtmlEntityDocumentConverter $documentConverter = null,
        HtmlEntityTableConverter $tableConverter = null,
        HtmlEntityTopListConverter $topListConverter = null
    ) {
        $this->imageConverter = $imageConverter ?: new HtmlEntityImageConverter();
        $this->embedConverter = $embedConverter ?: new HtmlEntityEmbedConverter();
        $this->linkConverter = $linkConverter ?: new HtmlEntityLinkConverter();
        $this->resourceConverter = $resourceConverter ?: new HtmlEntityResourceConverter();
        $this->documentConverter = $documentConverter ?: new HtmlEntityDocumentConverter();
        $this->tableConverter = $tableConverter ?: new HtmlEntityTableConverter();
        $this->topListConverter = $topListConverter ?: new HtmlEntityTopListConverter($this->imageConverter);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityLinkableInterface $entity
     * @param string $nodeText
     * @return \DOMElement|null
     */
    public function createLinkFromEntity(\DOMDocument $doc, EntityLinkableInterface $entity, $nodeText)
    {
        return $this->linkConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityImage $entity
     * @param string $nodeText
     * @return \DOMElement|null
     */
    public function createImageFromEntity(\DOMDocument $doc, EntityImage $entity, $nodeText)
    {
        return $this->imageConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityEmbed $entity
     * @param string $nodeText
     * @return \DOMElement
     */
    public function createEmbedFromEntity(\DOMDocument $doc, EntityEmbed $entity, $nodeText)
    {
        return $this->embedConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityDocument $entity
     * @param $nodeText
     * @return \DOMElement
     */
    public function createDocumentFromEntity(\DOMDocument $doc, EntityDocument $entity, $nodeText)
    {
        return $this->documentConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityResource $entity
     * @param string $nodeText
     * @return \DOMElement
     */
    public function createResourceFromEntity(\DOMDocument $doc, EntityResource $entity, $nodeText)
    {
        return $this->resourceConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityTable $entity
     * @param string $nodeText
     * @return \DOMElement|null
     */
    public function createTableFromEntity(\DOMDocument $doc, EntityTable $entity, $nodeText)
    {
        return $this->tableConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityTopList $entity
     * @param string $nodeText
     * @return \DOMElement|null
     */
    public function createTopListFromEntity(\DOMDocument $doc, EntityTopList $entity, $nodeText)
    {
        return $this->topListConverter->getEntityNode($doc, $entity, $nodeText);
    }

    /**
     * @param \DOMDocument $doc
     * @param EntityInterface $entity
     * @param string $nodeText
     * @return \DOMElement
     */
    public function getEntityNode(\DOMDocument $doc, EntityInterface $entity, $nodeText = null)
    {
        $classValue = [];
        if ($entity instanceof EntityPositionableInterface) {
            $classValue[] = 'align-' . $entity->getPosition();
        }
        // gstion des noeud resource
        if ($entity instanceof EntityResource) {
            $node = $this->createResourceFromEntity($doc, $entity, $nodeText);
        } // gstion des noeud embed
        elseif ($entity instanceof EntityEmbed) {
            $node = $this->createEmbedFromEntity($doc, $entity, $nodeText);
        } // gestion des noeuds documents
        elseif ($entity instanceof EntityDocument) {
            $node = $this->createDocumentFromEntity($doc, $entity, $nodeText);
        } // gestion des noeud images
        elseif ($entity instanceof EntityImage) {
            $imgNode = $this->createImageFromEntity($doc, $entity, $nodeText);
            $linkNode = $this->createLinkFromEntity($doc, $entity, $nodeText);
            if ($linkNode !== null) {
                $linkNode->appendChild($imgNode);
                $node = $linkNode;
            } else {
                $node = $imgNode;
            }
        } // gestion des noeud tablo
        elseif ($entity instanceof EntityTable) {
            $node = $this->createTableFromEntity($doc, $entity, $nodeText);
        } // gestion des noeud toplist
        elseif ($entity instanceof EntityTopList) {
            $node = $this->createTopListFromEntity($doc, $entity, $nodeText);
        } // gestion des noeud liens
        elseif ($entity instanceof EntityLinkableInterface) {
            $node = $this->createLinkFromEntity($doc, $entity, $nodeText);
        }

        // noeud par default
        if (!isset($node)) {
            $node = $doc->createElement('span');
        }

        // ajout des attributs commun
        if (!empty($classValue)) {
            $node->setAttribute('class', $node->getAttribute('class') . ' ' . implode(' ', $classValue));
        }
        return $node;
    }
}
