<?php

namespace Cms\DashParser\Serializer;

use Cms\DashParser\Block\BlockHeadline;
use Cms\DashParser\Block\BlockInterface;
use Cms\DashParser\Block\BlockTextAwareInterface;
use Cms\DashParser\Entity\EntityBlockInterface;
use Cms\DashParser\Entity\EntityInterface;
use Cms\DashParser\Entity\EntitySummary;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

use function Cms\DashParser\create_document;
use function Cms\DashParser\fix_amps;
use function Cms\DashParser\load_html;

/**
 * Class HtmlSerializer
 * @package Cms\DashParser\Serializer
 */
class HtmlSerializer implements SerializerInterface, LoggerAwareInterface
{
    use LoggerAwareTrait;

    /** @var string */
    private $replaceForSummary;

    /** @var bool */
    private $hasSummary;

    /** @var  EntityInterface[] */
    private $entityCollection;

    /** @var HtmlBlockConverter */
    private $blockConverter;

    /** @var HtmlEntityConverter */
    private $entityConverter;

    /** @var HtmlStyleConverter */
    private $styleConverter;

    /**
     * HtmlSerializer constructor.
     * @param array $entityCollection
     * @param HtmlEntityConverter $entitySerializer
     * @param HtmlBlockConverter $blockSerializer
     * @param HtmlStyleConverter $styleSerializer
     */
    public function __construct(
        array $entityCollection,
        HtmlEntityConverter $entitySerializer = null,
        HtmlBlockConverter $blockSerializer = null,
        HtmlStyleConverter $styleSerializer = null
    ) {
        $this->entityCollection = $entityCollection;
        $this->entityConverter = $entitySerializer ?: new HtmlEntityConverter();
        $this->blockConverter = $blockSerializer ?: new HtmlBlockConverter();
        $this->styleConverter = $styleSerializer ?: new HtmlStyleConverter();
        $this->logger = new NullLogger();
        $this->hasSummary = false;
    }


    /**
     * @param string $key
     * @return EntityInterface|null
     */
    public function getEntityByKey($key)
    {
        // check exist entity
        if (!isset($this->entityCollection[$key])) {
            return null;
        }
        return $this->entityCollection[$key];
    }

    /**
     * @param string $text
     * @param array $chunkPos
     * @param array $chunk
     * @return array
     */
    protected function chunkTextInto($text, array $chunkPos, array $chunk = [])
    {
        // découpe du texte
        $chunkPos = array_unique($chunkPos);
        sort($chunkPos);
        foreach ($chunkPos as $k => $offset) {
            if (!isset($chunkPos[$k + 1])) {
                break;
            }
            $pos = $chunkPos[$k + 1] - $offset;

            $textChunk = mb_substr($text, 0, $pos);
            //html_entities($textChunk);
            $chunk[$offset][] = $textChunk;

            $text = mb_substr($text, $pos);
        }

        ksort($chunk);
        return $chunk;
    }

    /**
     * @param string $nodeName
     * @param \Iterator|array $attrs
     * @param int $offsetBegin
     * @param int $length
     * @param array $chunk
     * @return array
     */
    protected function chunkForNodeInto(
        $nodeName,
        $attrs,
        $offsetBegin,
        $length,
        array $chunk = []
    ) {
        // construction des attribut de la balise
        $attrsHtml = '';
        foreach ($attrs as $name => $attr) {
            if (!empty($attr)) {
                $attrsHtml .= ' ' . $name;
                if (!\is_bool($attr)) {
                    $attrsHtml .= '="' . str_replace('"', '&quote;', $attr) . '"';
                }
            }
        }

        // ajout de la balise d'ouverture
        $offsetEnd = $offsetBegin + $length;
        if (!isset($chunk[$offsetBegin])) {
            $chunk[$offsetBegin] = [];
        }
        if (!isset($chunk[$offsetBegin][$length])) {
            $chunk[$offsetBegin][$length] = [];
        }
        $this->logger->debug(
            "open new node : $nodeName with attributes ($attrsHtml)",
            [
                'offsetBegin' => $offsetBegin,
                'offsetEnd' => $offsetEnd,
            ]
        );
        $chunk[$offsetBegin][$length][] = "<$nodeName$attrsHtml>";

        // si balise ouvrante/fermante on stop
        if ((int)$length === 0) {
            return $chunk;
        }

        // check si des balise se ferme avant moi pour evité les collisions
        ksort($chunk);
        $offsetEndInserted = false;
        foreach ($chunk as $offset => &$item) {
            // on demarre a mon offset de depart
            if ($offset <= $offsetBegin) {
                continue;
            }
            // on s'arrete a mon offset de fin
            if ($offset > $offsetEnd) {
                break;
            }
            if (!\is_array($item)) {
                continue;
            }
            foreach ($item as &$node) {
                $this->logger->debug(
                    "close node $nodeName before other node open : " . $node[0],
                    [
                        'offsetBegin' => $offsetBegin,
                        'offsetEnd' => $offsetEnd,
                    ]
                );
                array_unshift($node, "</$nodeName>");
                // on ajoute pas la dernière balise sinon on devrai la refermé après
                if ($offset < $offsetEnd) {
                    $this->logger->debug(
                        "open node after closed : $nodeName with attributes ($attrsHtml)",
                        [
                            'offsetBegin' => $offsetBegin,
                            'offsetEnd' => $offsetEnd,
                        ]
                    );
                    $node[] = "<$nodeName$attrsHtml>";
                } else {
                    $offsetEndInserted = true;
                }
            }
            unset($node);
        }
        unset($item);

        // ajout de la balise de fermeture
        if ($offsetEndInserted === false) {
            if (!isset($chunk[$offsetEnd])) {
                $chunk[$offsetEnd] = [];
            }
            if (!isset($chunk[$offsetEnd][$length])) {
                $chunk[$offsetEnd][$length] = [];
            }
            $this->logger->debug(
                "close node : $nodeName",
                [
                    'offsetBegin' => $offsetBegin,
                    'offsetEnd' => $offsetEnd,
                ]
            );
            $chunk[$offsetEnd][$length][] = "</$nodeName>";
        }
        return $chunk;
    }

    /**
     * @param string $contentHtml
     * @param array $styleRanges
     * @param array $chunk
     * @return array
     */
    protected function chunkWithStyle($contentHtml, array $styleRanges, array $chunk = [])
    {
        foreach ($styleRanges as $inlineStyle) {
            // detection de la balise
            $nodeName = $this->styleConverter->getStyleNodeName(
                $inlineStyle['style'],
                $contentHtml
            );
            if ($nodeName === null) {
                continue;
            }

            // calcul des offset
            $offsetBegin = $inlineStyle['offset'];
            $this->logger->debug(
                "new style detected : $nodeName",
                [
                    'contentHtml' => $contentHtml,
                    'offsetBegin' => $offsetBegin,
                ]
            );

            // ajoute un chunk de balise
            $chunk = $this->chunkForNodeInto(
                $nodeName,
                [],
                $offsetBegin,
                $inlineStyle['length'],
                $chunk
            );
        }

        return $chunk;
    }

    /**
     * @param string $contentHtml
     * @param array $entityRanges
     * @param array $chunk
     * @return array
     */
    protected function chunkWithEntity($contentHtml, array $entityRanges, array $chunk = [])
    {
        $doc = create_document();
        foreach ($entityRanges as $key => $entityRange) {
            // check exist entity
            $entity = $this->getEntityByKey($entityRange['key']);
            if ($entity === null) {
                continue;
            }

            // calcul des offset
            $offsetBegin = $entityRange['offset'];
            $offsetEnd = $offsetBegin + $entityRange['length'];
            $contentHtml = mb_substr($contentHtml, $offsetBegin, $entityRange['length']);
            $this->logger->debug(
                'new entity detected : ' . \get_class($entity),
                [
                    'contentHtml' => $contentHtml,
                    'offsetBegin' => $offsetBegin,
                    'offsetEnd' => $offsetEnd,
                ]
            );

            // detection de la balise
            $element = $this->entityConverter->getEntityNode($doc, $entity, $contentHtml);
            if ($element === null) {
                continue;
            }

            // is blockEntity
            if ($entity instanceof EntityBlockInterface) {
                $htmlElement = $doc->saveXML($element);
                // check si un sommaire est présent
                if ($entity instanceof EntitySummary) {
                    $this->hasSummary = true;
                    $this->replaceForSummary = $htmlElement;
                }
                if (!isset($chunk[$offsetEnd])) {
                    $chunk[$offsetEnd] = [];
                }
                $chunk[$offsetEnd][] = $htmlElement;
                continue;
            }

            // ajoute un chunk de balise
            $attrs = [];
            foreach ($element->attributes as $attr) {
                $attrs[$attr->nodeName] = $attr->nodeValue;
            }
            $chunk = $this->chunkForNodeInto(
                $element->nodeName,
                $attrs,
                $offsetBegin,
                $entityRange['length'],
                $chunk
            );
        }
        unset($doc);

        return $chunk;
    }

    /**
     * @param BlockInterface[] $blockCollection
     * @return string
     */
    public function serialize(array $blockCollection = [])
    {
        $html = '';
        $summary = [];
        foreach ($blockCollection as $block) {
            // check type
            if (!$block instanceof BlockInterface) {
                continue;
            }

            // creation du block
            $blockHtml = $this->blockConverter->createBlock($block, $html);
            // aucun bloc découvert on passe au suivant (ou saut de ligne)
            if ($blockHtml === null) {
                continue;
            }
            $this->logger->debug(
                'new block detected : ' . \get_class($block),
                [
                    'blockHtml' => $blockHtml,
                ]
            );

            // récuperation du sommaire
            if ($block instanceof BlockHeadline) {
                $summary[] = ['text' => $block->getText(), 'level' => $block->getLevel()];
            }

            // decoupe du block texte
            $contentHtml = '';
            $doc = create_document();
            if ($block instanceof BlockTextAwareInterface) {
                $contentHtml = $block->getText();
                if (in_array($contentHtml, ['image', 'embed', 'topList', 'summary'])) {
                    $contentHtml = '';
                } elseif ('' !== $blockHtml) {
                    // on charge le HTML en backup si aucun chunk ou que le chunk echoue
                    load_html($doc, $blockHtml);
                }
            }

            // chuck pour les styles
            $chunk = $this->chunkWithStyle($contentHtml, $block->getInlineStyleRanges());

            // chunk pour les entities
            $chunk = $this->chunkWithEntity($contentHtml, $block->getEntityRanges(), $chunk);

            // on check si on a un nombre de chunk suffisant
            if (count($chunk) > 0) {
                // ajoute un point sur le premier char
                if (!isset($chunk[0])) {
                    $chunk[0] = [];
                }
                // ajoute un point sur le dernier char
                $contentLength = mb_strlen($contentHtml);
                if (!isset($chunk[$contentLength])) {
                    $chunk[$contentLength] = [];
                }

                // découpe le texte en suivant les chunk
                $chunk = $this->chunkTextInto($contentHtml, array_keys($chunk), $chunk);

                // flatten array
                ksort($chunk);
                $chunk = new \RecursiveIteratorIterator(new \RecursiveArrayIterator($chunk));
                $chunk = iterator_to_array($chunk, false);
                $this->logger->debug(
                    'chunk for block is finalized : ' . \get_class($block),
                    [
                        'chunk' => $chunk,
                    ]
                );

                // mise a jour du block
                $contentHtml = nl2br(implode('', $chunk));
                if (!empty($contentHtml)) {
                    // on force la création d'un noeud parent pour fixé un potentiel ajout au moment du loadHTML
                    // cet ajout n'est util que si on veu faire du innerHTML par la suite sinon on prendra
                    // le noeud directement pour le rendu final
                    if ($doc->firstChild) {
                        $contentHtml = "<htmlfragment>$contentHtml</htmlfragment>";
                    }
                    $body = create_document();
                    if (!load_html($body, $contentHtml)) {
                        trigger_error('Invalid fragment : ' . $contentHtml, E_USER_WARNING);
                    } elseif ($doc->firstChild) {
                        // remplace le contenu HTML avec celui du chunk, on fait ca car on souhaite
                        // garder la configuration fourni au moment du createBlock, principalement le
                        // noeud de depart du block et ses potentiel attributs

                        // clean up childs
                        foreach ($doc->firstChild->childNodes as $childNode) {
                            $doc->firstChild->removeChild($childNode);
                        }
                        // import nodes
                        foreach ($body->firstChild->childNodes as $childNode) {
                            $importedNode = $doc->importNode($childNode, true);
                            $doc->firstChild->appendChild($importedNode);
                        }
                    } else {
                        // le document n'etait pas un block a l'origine on va donc utilisé
                        // le chunk dans son intégralité
                        $doc = $body;
                    }
                }
            }

            // ajoute le block
            if ($doc->firstChild) {
                $html .= $doc->saveHTML($doc->firstChild);
            }
        }

        // check si un sommaire est disponible
        if (!empty($summary) && $this->hasSummary) {
            // detection de la balise
            $entity = new EntitySummary($summary);
            $doc = create_document();
            $element = $this->entityConverter->getEntityNode($doc, $entity, '');
            if ($element !== null) {
                $html = str_replace($this->replaceForSummary, $doc->saveXML($element) ?: '', $html);
            } else {
                $html = str_replace($this->replaceForSummary, '', $html);
            }
        }

        return $html;
    }

    /**
     * @param string $html
     * @param int $offset
     * @deprecated
     */
    protected static function fixAmps(&$html, $offset = 0)
    {
        fix_amps($html, $offset);
    }

    /**
     * @return \DOMDocument
     * @deprecated
     */
    protected function createDomDocument(): \DOMDocument
    {
        return create_document();
    }

    /**
     * @param \DOMNode $element
     * @return string
     */
    protected function DOMinnerHTML(\DOMNode $element): string
    {
        $innerHTML = '';
        $children = $element->childNodes;
        foreach ($children as $child) {
            $innerHTML .= $element->ownerDocument->saveHTML($child);
        }
        return $innerHTML;
    }
}
