<?php

namespace Cms\DashParser\Serializer;

use Cms\DashParser\Entity\EntityInterface;
use Cms\DashParser\Entity\EntitySummary;

use function Cms\DashParser\create_document;
use function Cms\DashParser\load_html;
use function Cms\DashParser\slugify;

/**
 * Class HtmlEntitySummaryConverter
 * @package Cms\DashParser\Serializer
 */
class HtmlEntitySummaryConverter implements HtmlEntityConverterInterface
{

    /**
     * @var string
     */
    private $pictoUri;

    /**
     * @var int
     */
    private $maxLevel;

    /**
     * HtmlEntitySummaryConverter constructor.
     * @param string|null $pictoUri
     * @param int $maxLevel
     */
    public function __construct(string $pictoUri = null, int $maxLevel = 3)
    {
        $this->pictoUri = $pictoUri;
        $this->maxLevel = $maxLevel;
    }

    /**
     * @inheritDoc
     */
    public function getEntityNode(\DOMDocument $doc, EntityInterface $entity, $nodeText = null)
    {
        if (!$entity instanceof EntitySummary) {
            return null;
        }

        // placeholder for replacement
        if (!$entity->getSummary()) {
            return $doc->createElement('div', '##summary##');
        }

        // on vide le contenu
        $summary = $doc->createElement('div');
        $summary->setAttribute('class', 'summary-art');

        // creation du title avec le picto
        $pictoUri = $this->pictoUri;
        if ($pictoUri) {
            $summaryTitleHtml = <<<EOF
<div class="summary-title">
    <span class="pic">
        <svg width="25" height="18">
            <use xlink:href="$pictoUri"></use>
        </svg>
    </span>
    <span class="txt">Sommaire</span>
</div>
EOF;
        } else {
            $summaryTitleHtml = <<<EOF
<div class="summary-title">
    <span class="txt">Sommaire</span>
</div>
EOF;
        }

        $summaryTitle = create_document();
        if (load_html($summaryTitle, $summaryTitleHtml)) {
            $summaryTitle = $doc->importNode($summaryTitle->firstChild, true);
            $summary->appendChild($summaryTitle);
        }

        $chapter = ['currentLevel' => null, 'lists' => [$doc->createElement('ol')]];
        foreach ($entity->getSummary() as $summaryData) {
            if ($summaryData['level'] >= $this->maxLevel) {
                continue;
            }

            $summaryItemLink = $doc->createElement('a', $summaryData['text']);
            $summaryItemLink->setAttribute('href', '#hl_' . slugify($summaryData['text']));

            $summaryItem = $doc->createElement('li');
            $summaryItem->appendChild($summaryItemLink);

            if ($chapter['currentLevel'] === null) {
                $chapter['currentLevel'] = $summaryData['level'];
            } elseif ($chapter['currentLevel'] < $summaryData['level']) {
                array_unshift($chapter['lists'], $doc->createElement('ol'));
            } elseif ($chapter['currentLevel'] > $summaryData['level']) {
                $parentList = array_shift($chapter['lists']);
                $chapter['lists'][0]->appendChild($parentList);
            }
            $chapter['currentLevel'] = $summaryData['level'];

            $chapter['lists'][0]->appendChild($summaryItem);
        }
        if (count($chapter['lists']) > 1) {
            $parentList = array_shift($chapter['lists']);
            $chapter['lists'][0]->appendChild($parentList);
        }

        $summary->appendChild($chapter['lists'][0]);

        return $summary;
    }
}
