<?php

namespace POM;

/**
 * Class MapperSqlAbstract
 * @package POM
 */
abstract class MapperSqlAbstract extends MapperAbstract
{

    /**
     * @param string $value
     * @return string
     */
    abstract protected function quoteString($value);

    /**
     * Returns the SQL condition to identify this model
     * @param array $entities
     * @return array (SQL_WHERE, BINDINGS, IDMAP)
     */
    final protected function getEntityCondition(array $entities)
    {
        $primaries = $this->getEntityPrimaries();
        $entitiesFiltered = array_intersect_key($entities, array_flip($primaries));
        $condition = $this->getCondition($entitiesFiltered);
        array_push($condition, implode('-', $entitiesFiltered));
        return $condition;
    }

    /**
     * @param mixed $value
     * @return string
     */
    final protected function convertEntityValue($value)
    {
        // si serializable
        if ($value instanceof \JsonSerializable) {
            $value = json_encode($value);
        } elseif ($value instanceof \Serializable) {
            $value = $value->serialize();
        }
        return $value;
    }

    /**
     * Returns the SQL condition
     * @param array $entities
     * @param string $glue default ' AND '
     * @return array (SQL_WHERE, BINDINGS)
     */
    final protected function getCondition(array $entities, $glue = ' AND ')
    {
        // generation des valeur modifié
        $values = array_map([$this, 'convertEntityValue'], $entities);
        $keyEntities = array_keys($entities);

        // generation des placeholders
        $placeholders = array_combine($keyEntities, array_map(function ($key, $value) {
            $key = strtolower(str_replace('.', '_', $key));
            if (is_array($value)) {
                $keys = [];
                for ($i = 1; $i <= count($value); $i++) {
                    $keys[] = $key . '_' . $i;
                }
                return $keys;
            }
            return $key;
        }, $keyEntities, $values));

        // gestion de la chaine de conditions
        $conditions = array_map(function ($key, $placeholder, $value) {
            if (false === strpos($key, '`.`')) {
                $key = str_replace('.', '`.`', $key);
            }
            if (is_array($value)) {
                return sprintf("`%s` IN (:" . implode(', :', $placeholder) . ")", $key);
            } else {
                $operator = is_scalar($value)
                && (preg_match("@^%@", $value) === 1 || preg_match("@%$@", $value) === 1)
                    ? 'LIKE'
                    : '=';
                return sprintf("`%s` $operator :$placeholder", (string)$key);
            }
        }, $keyEntities, $placeholders, $values);

        // generation du ntableau de binding
        $bindings = [];
        foreach ($placeholders as $phName => $placeholder) {
            if (is_array($placeholder)) {
                foreach ($placeholder as $kph => $ph) {
                    $bindings[$ph] = $values[$phName][$kph];
                }
            } else {
                $bindings[$placeholder] = $values[$phName];
            }
        }
        $sql = implode($glue, $conditions);
        ksort($bindings);
        return [$sql, $bindings];
    }


    /**
     * Charge les données dans $object de l'element trouvé via son id,
     * renvoi true/false selon la réussite de la requete
     * @param mixed $id
     * @param DomainObjectInterface $object
     * @return bool
     */
    public function fetchById($id, DomainObjectInterface &$object)
    {
        if (!is_array($id)) {
            $id = array_combine($this->getEntityPrimaries(), [$id]);
        }
        list($where, $bindings) = $this->getEntityCondition($id);
        $query = 'SELECT * FROM `' . $this->getEntityTable() . '` WHERE ' . $where;
        // on ne sauvegarde les données que si elle sont rempli
        if (!($data = $this->service->fetchOne($query, $bindings))) {
            return false;
        }
        $this->populate($object, $data, true);
        return true;
    }

    /**
     * Supprime un element de la DB via son ID,
     * renvoi true/false selon la réussite de la requete
     * @param mixed $id
     * @return bool
     */
    public function removeById($id)
    {
        if (!is_array($id)) {
            $id = array_combine($this->getEntityPrimaries(), [$id]);
        }
        list($where, $bindings) = $this->getEntityCondition($id);
        $query = 'DELETE FROM `' . $this->getEntityTable() . '` WHERE ' . $where;
        $this->service->exec($query, $bindings);
        return true;
    }

    /**
     * @param DomainObjectInterface $object
     * @param array $bindings
     * @return null|string
     */
    final protected function getSaveQuery(DomainObjectInterface $object, &$bindings = [])
    {
        $entities = array_filter($object->getArrayCopy());
        $keys = array_keys($entities);
        $bindings = array_map([$this, 'convertEntityValue'], $entities);
        if (!empty($bindings)) {
            $query = sprintf('REPLACE INTO `%s` (`%s`) VALUES (:%s)',
                $this->getEntityTable(),
                implode('`, `', $keys),
                implode(', :', $keys)
            );
            return $query;
        }
        return null;
    }

    /**
     * Sauvegarde l'objet dans la DB, l'$objet est mis a jour selon les modif appliqué par la DB (insert: id...)
     * renvoi true/false selon la réussite de la requete
     * @param DomainObjectInterface $object
     * @return bool
     */
    public function save(DomainObjectInterface &$object)
    {
        $query = $this->getSaveQuery($object, $values);
        if ($query !== null) {
            $this->service->exec($query, $values);
            return true;
        }
        return $this->insert($object);
    }

    /**
     * @param DomainObjectInterface $object
     * @param array $bindings
     * @param bool $ignore
     * @return null|string
     */
    final protected function getInsertQuery(DomainObjectInterface $object, &$bindings = [], $ignore = false)
    {
        $entities = array_filter($object->getArrayCopy());
        $keys = array_keys($entities);
        $bindings = array_map([$this, 'convertEntityValue'], $entities);
        if (!empty($bindings)) {
            $query = sprintf('INSERT %sINTO `%s` (`%s`) VALUES (:%s)',
                $ignore ? "IGNORE " : '',
                $this->getEntityTable(),
                implode('`, `', $keys),
                implode(', :', $keys)
            );
            return $query;
        }
        return null;
    }

    /**
     * Insert l'objet dans la DB, l'$objet est mis a jour selon les modif appliqué par la DB (insert: id...)
     * si l'objet existe déjà on renvoi un exception
     * renvoi true/false selon la réussite de la requete
     * @param DomainObjectInterface $object
     * @return bool
     */
    public function insert(DomainObjectInterface &$object)
    {
        $query = $this->getInsertQuery($object, $values);
        if ($query !== null && $this->service->exec($query, $values, $insertId)) {
            // insert de la clé
            $object[$this->getEntityPrimaries()[0]] = $insertId;
            return true;
        }
        return false;
    }

    /**
     * @param DomainObjectInterface $object
     * @param array $bindings
     * @return null|string
     */
    final protected function getUpdateQuery(DomainObjectInterface $object, &$bindings = [])
    {
        list($condition, $bindings) = $this->getEntityCondition($object->getArrayCopy());

        if (!empty($condition) && !empty($bindings)) {
            $primaries = $this->getEntityPrimaries();
            //$entities = array_filter($object->getArrayCopy(true));
            $entities = array_diff_key($object->getArrayCopy(true), array_flip($primaries));
            list($update, $updateBindings) = $this->getCondition($entities, ', ');
            $bindings = array_merge($updateBindings, $bindings);

            $query = sprintf('UPDATE `%s` SET %s WHERE %s',
                $this->getEntityTable(),
                $update,
                $condition);

            return $query;
        }

        return null;
    }

    /**
     * Update l'objet dans la DB, l'$objet est mis a jour selon les modif appliqué par la DB (insert: id...)
     * si l'objet n'existe pas déjà on renvoi un exception
     * renvoi true/false selon la réussite de la requete
     * @param DomainObjectInterface $object
     * @return mixed
     */
    public function update(DomainObjectInterface &$object)
    {
        $query = $this->getUpdateQuery($object, $bindings);
        if ($query !== null) {
            $this->service->exec($query, $bindings);
            return true;
        }
        return false;
    }

    /**
     * Supprime l'objet de la DB, si l'objet n'existe pas déjà on renvoi un exception
     * renvoi true/false selon la réussite de la requete
     * @param DomainObjectInterface $object
     * @return mixed
     */
    public function remove(DomainObjectInterface $object)
    {
        $id = [];
        foreach ($this->getEntityPrimaries() as $primaryKey) {
            $id[$primaryKey] = $object[$primaryKey];
        }
        return empty($id) ? false : $this->removeById($id);
    }

}