<?php

namespace POM\Service;

/**
 * Class CursorPdo
 * @package POM\Service
 */
class CursorPdo implements \SeekableIterator, \Countable
{

    /**
     * @var \PDOStatement
     */
    private $_statement = null;

    /**
     * @var array
     */
    private $_row = false;

    /**
     * @var int
     */
    private $_position = -1;

    /**
     * @var int
     */
    private $_total = null;

    /**
     * @var array
     */
    private $_callback;

    /**
     * @var array
     */
    private $_bindings = [];

    /**
     * @var AdapterPdoAbstract
     */
    private $_pdoAdapter;

    /**
     * @var string
     */
    private $_query;

    /**
     * @var bool
     */
    private $_seekable;

    /**
     * @var int
     */
    private $_limit = null;

    /**
     * @var int
     */
    private $_offset = 0;

    /**
     * @var int
     */
    private $_fetchMode;


    /**
     * @param AdapterPdoAbstract $pdoAdapter
     * @param string $query
     * @param array $bindings
     * @param int $fetchMode
     */
    public function __construct(
        AdapterPdoAbstract $pdoAdapter,
        $query,
        array $bindings = [],
        $fetchMode = \PDO::FETCH_ASSOC
    ) {
        // charge les données
        $this->_fetchMode = $fetchMode;
        $this->_bindings = $bindings;
        $this->_pdoAdapter = $pdoAdapter;
        $this->_query = $query;
        // check si on pe seek via la limit mysql ou si on seek a la main
        $this->_seekable = false;
        if (!preg_match('@LIMIT +\d+, *\d+@i', $query)) {
            $this->_seekable = true;
        }
        // execution de la requete pour obtenir le nombre total de ligne
        $this->executeTotal();
    }

    /**
     * ferme le curseur
     */
    public function __destruct()
    {
        if ($this->_statement) {
            $this->_statement->closeCursor();
        }
    }

    /**
     * @param callable $callback
     * @param array $args
     */
    public function setRowHandler(callable $callback, array $args = [])
    {
        $this->_callback = [$callback, $args];
    }


    /**
     * Return the current element
     * @link http://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     */
    public function current()
    {
        // si on a pas encore avancé...
        if (!$this->_row) {
            $this->next();
        }
        //error_log("[" . $this->_query . "] ask for current at position $this->_position" . ' (' . gettype($this->_row) . ')');
        // execution du handler de ligne
        if ($this->_row !== false && $this->_callback) {
            return call_user_func_array($this->_callback[0], array_merge([$this->_row], $this->_callback[1]));
        }
        return $this->_row;
    }

    /**
     * Charge la valeur du total
     * @return void
     */
    public function executeTotal()
    {
        if ($this->_total === null) {
            $query = 'SELECT COUNT(*) AS Total FROM (' . $this->_query . ') as t';
            $this->_statement = $this->_pdoAdapter->getStatementForQuery($query);
            $this->_pdoAdapter->bindValue($this->_statement, $this->_bindings);
            $this->_statement->execute();
            $this->_total = $this->_statement->fetchAll()[0]['Total'];
            //error_log("[" . $query . "] execute total for query " . $this->_total);
            $this->_statement = null;
        }
    }

    /**
     * Execute la requete réelement sur le serveur
     * @return void
     */
    public function execute()
    {
        $query = $this->_query;
        if ($this->isSeekable() && $this->_total !== null) {
            $query .= " LIMIT $this->_offset," . ($this->_limit ?: ($this->_total - $this->_position));
        }
        $this->_statement = $this->_pdoAdapter->getStatementForQuery($query);
        $this->_pdoAdapter->bindValue($this->_statement, $this->_bindings);
        $this->_statement->execute();
        //error_log("[" . $query . "] execute query " . $this->_statement->rowCount());
    }

    /**
     * Move forward to next element
     * @link http://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     */
    public function next()
    {
        if ($this->_total === 0) {
            $this->_row = false;
            return null;
        }
        if (!$this->_statement) {
            $this->execute();
        }

        $this->_row = $this->_statement->fetch(
            $this->_fetchMode
        );

        $this->_position++;
        //error_log("[" . $this->_statement->queryString . "] next pdo cursor to " . $this->_position . ' (' . gettype($this->_row) . ')');
    }

    /**
     * Return the key of the current element
     * @link http://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     */
    public function key()
    {
        return $this->_position;
    }

    /**
     * Checks if current position is valid
     * @link http://php.net/manual/en/iterator.valid.php
     * @return boolean The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     */
    public function valid()
    {
        $valid = $this->_statement ? $this->_row !== false : (
        $this->_total ? $this->_position < $this->_total : false
        );
        //error_log("[" . $this->_query . "] check validity " . $this->_position . ' ' . $valid . ' (' . gettype($this->_row) . ')');
        return $valid;
    }

    /**
     * Rewind the Iterator to the first element
     * @link http://php.net/manual/en/iterator.rewind.php
     * @throws \OutOfBoundsException
     * @return void Any returned value is ignored.
     */
    public function rewind()
    {
        //error_log("[" . $this->_query . "] rewind pdo cursor from " . $this->_position);
        if ($this->_position > 0) {
            if ($this->_statement) {
                $this->_statement->closeCursor();
                $this->_statement = null;
            }
            $this->_position = -1;
        }
    }

    /**
     * @inheritdoc
     */
    public function count()
    {
        return $this->_total;
    }

    /**
     * Seeks to a position
     * @link http://php.net/manual/en/seekableiterator.seek.php
     * @param int $position The position to seek to.
     * @return void
     */
    public function seek($position)
    {
        //error_log("[" . $this->_query . "] seek pdo cursor to $position");
        if ($this->isSeekable()) {
            if ($this->_statement && $this->_offset !== (int)$position) {
                $this->_statement->closeCursor();
                $this->_statement = null;
            }
            $this->_offset = (int)$position;
            $this->_position = (int)$position;
        } else {
            while ($this->_position < $position) {
                $this->next();
            }
        }
    }

    /**
     * @param int $position
     * @return $this
     */
    public function offset($position)
    {
        $this->seek($position);
        return $this;
    }

    /**
     * @param int $byPage
     * @return $this
     */
    public function limit($byPage)
    {
        $this->_limit = $byPage;
        return $this;
    }

    /**
     * @return \PDOStatement
     */
    public function getStatement()
    {
        return $this->_statement;
    }

    /**
     * @return boolean
     */
    public function isSeekable()
    {
        return $this->_seekable;
    }
}