<?php

namespace POM\Service;

use POM\IdentityMap;

/**
 * Class AdapterPdoAbstract
 * @package POM\Service
 */
abstract class AdapterPdoAbstract extends AdapterAbstract
{

    /**
     * @var array
     */
    protected $dbAccess;

    /**
     * @var \PDO
     */
    protected $dbHandler;

    /**
     * @var IdentityMap
     */
    protected $stmtMap;

    /**
     * @param string $dsn
     * @param string $user
     * @param string $pass
     * @param array $opts
     */
    public function __construct($dsn, $user, $pass, array $opts = [])
    {
        $this->dbAccess['dsn'] = $dsn;
        $this->dbAccess['user'] = $user;
        $this->dbAccess['pass'] = $pass;
        $this->dbAccess['opts'] = array_merge(array(
            \PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION,
        ), $opts);
        // on utilise une identity map pour stocké les statements afin de les réutiliser
        $this->stmtMap = new IdentityMap();
    }

    /**
     * detruit la connection actuellement ouverte
     */
    public function __destruct()
    {
        $this->deconnect();
    }

    /**
     * Déconnection de la DB
     */
    public function deconnect()
    {
        unset($this->dbHandler);
    }

    /**
     * Connection de la DB, si erreur Exception est envoyé, sinon true si connexion requise,
     * false si déja effectué
     * @return bool
     */
    public function connect()
    {
        if (!$this->dbHandler instanceof \PDO) {
            $this->dbHandler = new \PDO(
                $this->dbAccess['dsn'],
                $this->dbAccess['user'],
                $this->dbAccess['pass'],
                $this->dbAccess['opts']
            );
            return true;
        }
        return false;
    }

    /**
     * @param string $string
     * @return string
     */
    public function quoteString($string)
    {
        if (!is_string($string)) {
            throw new \InvalidArgumentException("this function can only quote string.");
        }
        return $this->getDbHandler()->quote($string);
    }

    /**
     * Renvoi l'instance de DBHandler
     * @return \PDO
     */
    public function getDbHandler()
    {

        $this->connect();
        return $this->dbHandler;
    }

    /**
     * Renvoi le statement associé a une query
     * @param string $query
     * @return \PDOStatement
     */
    public function getStatementForQuery($query)
    {
        // on verifie que le statement n'est pas deja présent
        if (!$this->stmtMap->hasId($query)) {
            $stmt = $this->getDbHandler()->prepare($query, [
                \PDO::ATTR_CURSOR => \PDO::CURSOR_SCROLL
            ]);
            $this->stmtMap->storeObject($query, $stmt);
        } else {
            $stmt = $this->stmtMap->getObject($query);
        }
        return $stmt;
    }

    /**
     * Assign a un statement les bind
     * @param \PDOStatement $stmt
     * @param array $bind
     * @return void
     */
    public function bindValue(\PDOStatement &$stmt, array $bind)
    {
        $bind = array_map(function ($value, $key) {
            $sValueType = \PDO::PARAM_STR;
            if (is_int($value)) {
                $value = (int)$value;
            } elseif ($value instanceof \DateTime) {
                $value = $value->format("Y-m-d H:i:s");
            }
            if (is_bool($value)) {
                $sValueType = \PDO::PARAM_BOOL;
            } elseif (is_null($value)) {
                $sValueType = \PDO::PARAM_NULL;
            } elseif (is_int($value)) {
                $sValueType = \PDO::PARAM_INT;
            }
            return [(is_numeric($key) ? $key + 1 : $key), (string)$value, $sValueType];
        }, $bind, array_keys($bind));

        // lecture du binding
        foreach ($bind as $set) {
            list($k, $v, $type) = $set;
            $stmt->bindValue($k, $v, $type);
        }
    }


    /**
     * @inheritdoc
     * @return CursorPdo
     */
    public function fetch($query, array $bind = array())
    {
        $cursor = new CursorPdo($this, $query, $bind);
        return $cursor;
    }

    /**
     * @inheritdoc
     * @return CursorPdo
     */
    public function fetchColumn($column, $query, array $bind = array())
    {
        $cursor = new CursorPdo($this, $query, $bind, \PDO::FETCH_COLUMN);
        $cursor->getStatement()->setFetchMode(\PDO::FETCH_COLUMN, $column);
        return $cursor;
    }

    /**
     * Execute une liste de requete dans une transaction et renvoi le nombre total de ligne affecté
     * si $lastInsertId est fourni il est rempli avec le dernier ID inséré
     * @param array $queryList [ ['SQL QUERY', ['PARAM'=>'VALUE', ...]], 'SQL QUERY', ... ]
     * @param int $lastInsertId
     * @throws \Exception
     * @throws \PDOException
     * @return int
     */
    public function execInTransaction(array $queryList, &$lastInsertId = null)
    {
        try {
            $rowCount = 0;
            $this->getDbHandler()->beginTransaction();
            foreach ($queryList as $queryParam) {
                if (is_string($queryParam)) {
                    $rowCount += $this->exec($queryParam, []);
                } elseif (is_array($queryParam)) {
                    $rowCount += $this->exec($queryParam[0], $queryParam[1]);
                }
            }
            $this->getDbHandler()->commit();
            $lastInsertId = $this->getDbHandler()->lastInsertId();
        } catch (\PDOException $e) {
            throw $e;
        }
        return $rowCount;
    }

    /**
     * Effectue une requete et renvoi le nobre de ligne affecté,
     * si $lastInsertId est fourni il est rempli avec le dernier ID inséré
     * @param string $query
     * @param array $bind
     * @param int $lastInsertId
     * @return int
     */
    public function exec($query, array $bind = array(), &$lastInsertId = null)
    {
        $stmt = $this->getStatementForQuery($query);
        !$bind ?: $this->bindValue($stmt, $bind);
        if ($result = $stmt->execute()) {
            $lastInsertId = $this->getDbHandler()->lastInsertId();
            return $stmt->rowCount();
        }
        return 0;
    }

}