<?php

namespace POM;

/**
 * Class DomainObjectAbstract
 * @package POM
 */
abstract class DomainObjectAbstract implements DomainObjectInterface
{

    /** @var array */
    private $editableTypesProperties = [];

    /** @var array */
    private $modifiedOffsets = [];

    /**
     * Renvoi la liste des propriétés editable,
     * par default les propriétés protected sont toutes editable
     * @return array
     */
    public function getEditableProperties()
    {
        return array_keys($this->getEditableTypesProperties());
    }

    /**
     * Renvoi une liste de callable associé a la value pour les valeur de properties,
     * par default les propriétés protected sont toutes editable
     * @return callable[]
     */
    public function getEditableTypesProperties()
    {
        if ($this->editableTypesProperties) {
            return $this->editableTypesProperties;
        }
        $aProperties = (new \ReflectionClass($this))->getProperties(\ReflectionProperty::IS_PROTECTED);
        foreach ($aProperties as $property) {
            $callableForType = null;
            try {
                // on check le type déclaré
                if (preg_match(
                        '#@var\s+([^\s]+)#i',
                        $property->getDocComment(),
                        $matches
                    ) === 1 && !empty($matches[1])
                ) {
                    $detectType = $matches[1];
                    $callableForType = TypeFactory::getCallableForType($detectType, $this);
                }
            } catch (\ReflectionException $e) {
            }
            $this->editableTypesProperties[$property->getName()] = $callableForType;
        }
        return $this->editableTypesProperties;
    }

    /**
     * Charge le model a partir d'un tableau de données
     * @param array $dataset [attribut => 'valeur', ...]
     * @param bool $origin
     * @return $this
     */
    public function populate(array $dataset, $origin = false)
    {
        if (!empty($dataset)) {
            foreach ($this->getEditableProperties() as $propName) {
                if (isset($dataset[$propName])) {
                    $this->propertySet($propName, $dataset[$propName]);
                }
            }
        }
        if ($origin === true) {
            $this->modifiedOffsets = [];
        }
        return $this;
    }

    /**
     * Renvoi une copie sous forme de tableau des propriété du model avec leur valeur
     * si $modified_only est true on renvoi uniquement les valeur modifiers depuis le dernier
     * chargement
     * @param bool $modified_only
     * @return array [attribut => 'valeur', ...]
     */
    public function getArrayCopy($modified_only = false)
    {
        $properties = $this->getEditableProperties();
        if ($modified_only === true) {
            $properties = $this->modifiedOffsets;
        }
        $aToArrayCopy = [];
        foreach ($properties as $propName) {
            $aToArrayCopy[$propName] = $this->propertyGet($propName);
        }
        return $aToArrayCopy;
    }


    /**
     * Permet de valider l'intégrité du model
     * @return bool
     */
    public function validate()
    {
        return true;
    }

    /**
     * @param string $name
     * @return bool
     */
    public function propertyExists($name) : bool
    {
        return in_array($name, $this->getEditableProperties());
    }

    /**
     * @param string $name
     * @param mixed $value
     */
    public function propertySet($name, $value)
    {
        if ($this->propertyExists($name)) {
            if ($value !== null) {
                $aEditableTypes = $this->getEditableTypesProperties();
                if (!empty($aEditableTypes[$name]) && is_callable($aEditableTypes[$name])) {
                    $value = call_user_func($aEditableTypes[$name], $value);
                }
            }
            if ($this->$name !== $value && !in_array($name, $this->modifiedOffsets)) {
                $this->modifiedOffsets[] = $name;
            }
            $this->$name = $value;
        }
    }

    /**
     * @param string $name
     * @return mixed|null
     */
    public function propertyGet($name)
    {
        if ($this->propertyExists($name)) {
            return $this->$name;
        }
        trigger_error("La propriété $name n'existe pas", E_USER_NOTICE);
        return null;
    }

    /**
     * @param string $name
     */
    public function propertyUnset($name)
    {
        if ($this->offsetExists($name)) {
            $this->$name = null;
            // supprime la clé du tableau de modif
            if (false !== ($key = array_search($name, $this->modifiedOffsets))) {
                unset($this->modifiedOffsets[$key]);
            }
        }
    }

    /**
     * Permet la verification d'une propriété via l'objet comme si les attribut était public :
     *    isset($model->attribut);
     * @param string $offset
     * @return bool
     */
    public function __isset($offset)
    {
        return $this->propertyExists($offset);
    }

    /**
     * permet l'edition de propriétés via l'objet comme si les attribut était public :
     *    $model->attribut = 'new value';
     * @param string $offset
     * @param mixed $value
     */
    public function __set($offset, $value)
    {
        $this->propertySet($offset, $value);
    }

    /**
     * permet l'accès aux propriétés via l'objet comme si les attribut était public :
     *    echo $model->attribut;
     * @param string $offset
     * @return mixed
     */
    public function __get($offset)
    {
        return $this->propertyGet($offset);
    }

    /**
     * Permet la suppression d'une propriété via l'objet comme si les attribut était public :
     *    unset($model->attribut);
     * @param string $offset
     */
    public function __unset($offset)
    {
        $this->propertyUnset($offset);
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetexists.php
     * @param string $offset
     * @return bool
     */
    public function offsetExists($offset)
    {
        return $this->propertyExists($offset);
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetget.php
     * @param string $offset
     * @return mixed
     */
    public function offsetGet($offset)
    {
        return $this->propertyGet($offset);
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetset.php
     * @param string $offset
     * @param mixed $value
     */
    public function offsetSet($offset, $value)
    {
        $this->propertySet($offset, $value);
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetunset.php
     * @param string $offset
     */
    public function offsetUnset($offset)
    {
        $this->propertyUnset($offset);
    }


    /**
     * @see http://php.net/manual/en/iteratoraggregate.getiterator.php
     * @return \Traversable
     */
    public function getIterator()
    {
        return new \ArrayIterator($this->getArrayCopy());
    }

    /**
     * @return array
     */
    public function getModifiedProperties() : array
    {
        return $this->modifiedOffsets;
    }
}
