<?php

namespace POM\Service;

/**
 * Class CursorPdo
 * @package POM\Service
 */
class CursorPdo implements \SeekableIterator, \Countable
{

    /**
     * @var \PDOStatement
     */
    private $statement = null;

    /**
     * @var array
     */
    private $row = false;

    /**
     * @var int
     */
    private $position = - 1;

    /**
     * @var int
     */
    private $total = null;

    /**
     * @var array
     */
    private $callback;

    /**
     * @var array
     */
    private $bindings = [];

    /**
     * @var AdapterPdoAbstract
     */
    private $pdoAdapter;

    /**
     * @var string
     */
    private $query;

    /**
     * @var bool
     */
    private $seekable;

    /**
     * @var int
     */
    private $limit = null;

    /**
     * @var int
     */
    private $offset = 0;

    /**
     * @var int
     */
    private $fetchMode;


    /**
     * @param AdapterPdoAbstract $pdoAdapter
     * @param string $query
     * @param array $bindings
     * @param int $fetchMode
     */
    public function __construct(
        AdapterPdoAbstract $pdoAdapter,
        $query,
        array $bindings = [],
        $fetchMode = \PDO::FETCH_ASSOC
    ) {
        // charge les données
        $this->fetchMode = $fetchMode;
        $this->bindings = $bindings;
        $this->pdoAdapter = $pdoAdapter;
        $this->query = $query;
        // check si on pe seek via la limit mysql ou si on seek a la main
        $this->seekable = false;
        if (!preg_match('@LIMIT +\d+, *\d+@i', $query)) {
            $this->seekable = true;
        }
    }

    /**
     * ferme le curseur
     */
    public function __destruct()
    {
        if ($this->statement) {
            $this->statement->closeCursor();
        }
    }

    /**
     * @param callable $callback
     * @param array $args
     */
    public function setRowHandler(callable $callback, array $args = [])
    {
        $this->callback = [$callback, $args];
    }


    /**
     * Return the current element
     * @link http://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     */
    public function current()
    {
        // si on a pas encore avancé...
        if (!$this->row) {
            $this->next();
        }
        //error_log("[" . $this->_query . "] ask for current at position $this->_position" . ' (' . gettype($this->_row) . ')');
        // execution du handler de ligne
        if ($this->row !== false && $this->callback) {
            return call_user_func_array(
                $this->callback[0],
                array_merge([$this->row], $this->callback[1])
            );
        }
        return $this->row;
    }

    /**
     * Charge la valeur du total
     * @return void
     */
    public function executeTotal()
    {
        if ($this->total === null) {
            // TODO : supprime de la requete la partie avec order by
            $query = $this->query;
            $query = 'SELECT COUNT(*) AS Total FROM (' . $query . ') as t';
            $statement = $this->pdoAdapter->getStatementForQuery($query);
            $this->pdoAdapter->bindValue($statement, $this->bindings);
            $statement->execute();
            $statement->setFetchMode(\PDO::FETCH_COLUMN, 0);
            $this->total = $statement->fetch();
            $statement->closeCursor();
            //error_log("[" . $query . "] execute total for query " . $this->_total);
        }
    }

    /**
     * Execute la requete réelement sur le serveur
     * @return void
     */
    public function execute()
    {
        $query = $this->query;
        if ($this->isSeekable()) {
            if (!$this->limit) {
                $this->executeTotal();
            }
            $query .= " LIMIT $this->offset," . ($this->limit ?: ($this->total - $this->position));
        }
        $this->statement = $this->pdoAdapter->getStatementForQuery($query);
        $this->pdoAdapter->bindValue($this->statement, $this->bindings);
        $this->statement->execute();
        //error_log("[" . $query . "] execute query " . $this->_statement->rowCount());
    }

    /**
     * Move forward to next element
     * @link http://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     */
    public function next()
    {
        if (!$this->statement) {
            $this->execute();
        }

        if (!$this->statement || $this->statement->rowCount() === 0) {
            $this->row = false;
        } else {
            $this->row = $this->statement->fetch(
                $this->fetchMode,
                \PDO::FETCH_ORI_NEXT
            );
            $this->position ++;
        }
        //error_log("[" . $this->_statement->queryString . "] next pdo cursor to " . $this->_position . ' (' . gettype($this->_row) . ')');
    }

    /**
     * Return the key of the current element
     * @link http://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     */
    public function key()
    {
        return $this->position;
    }

    /**
     * Checks if current position is valid
     * @link http://php.net/manual/en/iterator.valid.php
     * @return boolean The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     */
    public function valid()
    {
        if (!$this->statement) {
            $this->executeTotal();
        }
        $valid = $this->statement ? $this->row !== false : (
        $this->total ? $this->position < $this->total : false
        );
        //error_log("[" . $this->_query . "] check validity " . $this->_position . ' ' . $valid . ' (' . gettype($this->_row) . ')');
        return $valid;
    }

    /**
     * Rewind the Iterator to the first element
     * @link http://php.net/manual/en/iterator.rewind.php
     * @throws \OutOfBoundsException
     * @return void Any returned value is ignored.
     */
    public function rewind()
    {
        //error_log("[" . $this->_query . "] rewind pdo cursor from " . $this->_position);
        if ($this->position > 0) {
            if ($this->statement) {
                $this->statement->closeCursor();
                $this->statement = null;
            }
            $this->position = - 1;
        }
    }

    /**
     * @inheritdoc
     */
    public function count()
    {
        $this->executeTotal();
        return $this->total;
    }

    /**
     * Seeks to a position
     * @link http://php.net/manual/en/seekableiterator.seek.php
     * @param int $position The position to seek to.
     * @return void
     */
    public function seek($position)
    {
        //error_log("[" . $this->_query . "] seek pdo cursor to $position");
        if ($this->isSeekable()) {
            if ($this->statement && $this->offset !== (int)$position) {
                $this->statement->closeCursor();
                $this->statement = null;
            }
            $this->offset = (int)$position;
            $this->position = (int)$position;
        } else {
            while ($this->position < $position) {
                $this->next();
            }
        }
    }

    /**
     * @param int $position
     * @return $this
     */
    public function offset($position)
    {
        $this->seek($position);
        return $this;
    }

    /**
     * @param int $byPage
     * @return $this
     */
    public function limit($byPage)
    {
        $this->limit = $byPage;
        return $this;
    }

    /**
     * @return \PDOStatement
     */
    public function getStatement()
    {
        return $this->statement;
    }

    /**
     * @return boolean
     */
    public function isSeekable()
    {
        return $this->seekable;
    }
}
