<?php

namespace POM;

/**
 * Class DomainObjectAbstract
 * @package POM
 */
abstract class DomainObjectAbstract implements DomainObjectInterface
{

    /** @var array */
    private $typesProperties = [];

    /** @var array */
    private $readonlyProperties = [];

    /** @var array */
    private $modifiedOffsets = [];

    /**
     * @return \ReflectionProperty[]
     */
    private function getProtectedProperties() : array
    {
        try {
            $reflect = new \ReflectionClass($this);
            return $reflect->getProperties(\ReflectionProperty::IS_PROTECTED);
        } catch (\ReflectionException $e) {
        }
        return [];
    }

    /**
     * Renvoi la liste des propriétés editable,
     * par default les propriétés protected sont toutes editable
     * @return array
     */
    public function getEditableProperties() : array
    {
        return array_keys($this->getTypesProperties());
    }

    /**
     * @deprecated
     * @return array
     */
    public function getEditableTypesProperties() : array
    {
        return $this->getTypesProperties();
    }

    /**
     * @return array
     */
    public function getReadonlyProperties() : array
    {
        if ($this->readonlyProperties) {
            return $this->readonlyProperties;
        }
        $properties = $this->getProtectedProperties();
        foreach ($properties as $property) {
            // parse le comment pour trouver le @relation
            if (preg_match('/@readonly/', $property->getDocComment())) {
                $this->readonlyProperties[] = $property->getName();
            }
        }
        return $this->readonlyProperties;
    }

    /**
     * Renvoi une liste de callable associé a la value pour les valeur de properties,
     * par default les propriétés protected sont toutes editable
     * @return callable[]
     */
    public function getTypesProperties() : array
    {
        if ($this->typesProperties) {
            return $this->typesProperties;
        }
        $properties = $this->getProtectedProperties();
        foreach ($properties as $property) {
            $callableForType = null;
            try {
                // on check le type déclaré
                if (preg_match(
                        '#@var\s+(\S+)#i',
                        $property->getDocComment(),
                        $matches
                    ) === 1 && !empty($matches[1])
                ) {
                    $detectType = $matches[1];
                    $callableForType = TypeFactory::getCallableForType($detectType, $this);
                }
            } catch (\ReflectionException $e) {
            }
            $this->typesProperties[$property->getName()] = $callableForType;
        }
        return $this->typesProperties;
    }

    /**
     * Charge le model a partir d'un tableau de données
     * @param array $dataset [attribut => 'valeur', ...]
     * @param bool $origin
     * @return $this
     */
    public function populate(array $dataset, $origin = false)
    {
        if (!empty($dataset)) {
            foreach ($this->getEditableProperties() as $propName) {
                if (isset($dataset[$propName])) {
                    $this->propertySet($propName, $dataset[$propName]);
                }
            }
        }
        if ($origin === true) {
            $this->modifiedOffsets = [];
        }
        return $this;
    }

    /**
     * Renvoi une copie sous forme de tableau des propriété du model avec leur valeur
     * si $modified_only est true on renvoi uniquement les valeur modifiers depuis le dernier
     * chargement
     * @param bool $modified_only
     * @return array [attribut => 'valeur', ...]
     */
    public function getArrayCopy($modified_only = false)
    {
        $properties = $this->getEditableProperties();
        if ($modified_only === true) {
            $properties = $this->modifiedOffsets;
        }
        $toArrayCopy = [];
        foreach ($properties as $propName) {
            $toArrayCopy[$propName] = $this->propertyGet($propName);
        }
        return $toArrayCopy;
    }


    /**
     * Permet de valider l'intégrité du model
     * @return bool
     */
    public function validate()
    {
        return true;
    }

    /**
     * @param string $name
     * @return bool
     */
    public function isReadonlyProperty(string $name) : bool
    {
        return in_array($name, $this->getReadonlyProperties(), false);
    }

    /**
     * @param string $name
     * @return bool
     */
    public function propertyExists($name) : bool
    {
        return in_array($name, $this->getEditableProperties(), false);
    }

    /**
     * @param string $name
     * @return callable|null
     */
    protected function getPropertySetter(string $name)
    {
        $aEditableTypes = $this->getTypesProperties();
        if (isset($aEditableTypes[$name]) && is_callable($aEditableTypes[$name])) {
            return $aEditableTypes[$name];
        }
        return null;
    }

    /**
     * @param string $name
     * @param mixed|null $value
     * @return mixed
     */
    final protected function usePropertySetter(string $name, $value = null)
    {
        if ($value !== null
            && null !== ($setter = $this->getPropertySetter($name))
        ) {
            /** @noinspection VariableFunctionsUsageInspection */
            return call_user_func($setter, $value);
        }
        return $value;
    }

    /**
     * @param string $name
     * @param mixed $value
     */
    public function propertySet($name, $value)
    {
        if ($this->propertyExists($name)) {
            $value = $this->usePropertySetter($name, $value);
            if ($this->$name !== $value && !in_array($name, $this->modifiedOffsets, false)) {
                $this->modifiedOffsets[] = $name;
            }
            $this->$name = $value;
        }
    }

    /**
     * @param string $name
     * @return mixed|null
     */
    public function propertyGet($name)
    {
        if ($this->propertyExists($name)) {
            return $this->$name;
        }
        trigger_error("La propriété $name n'existe pas");
        return null;
    }

    /**
     * @param string $name
     */
    public function propertyUnset($name)
    {
        if ($this->offsetExists($name)) {
            $this->$name = null;
            // supprime la clé du tableau de modif
            if (false !== ($key = array_search($name, $this->modifiedOffsets, false))) {
                unset($this->modifiedOffsets[$key]);
            }
        }
    }

    /**
     * Permet la verification d'une propriété via l'objet comme si les attribut était public :
     *    isset($model->attribut);
     * @param string $offset
     * @return bool
     */
    public function __isset($offset)
    {
        return $this->propertyExists($offset);
    }

    /**
     * permet l'edition de propriétés via l'objet comme si les attribut était public :
     *    $model->attribut = 'new value';
     * @param string $offset
     * @param mixed $value
     */
    public function __set($offset, $value)
    {
        if (!$this->isReadonlyProperty($offset)) {
            $this->propertySet($offset, $value);
        }
    }

    /**
     * permet l'accès aux propriétés via l'objet comme si les attribut était public :
     *    echo $model->attribut;
     * @param string $offset
     * @return mixed
     */
    public function __get($offset)
    {
        return $this->propertyGet($offset);
    }

    /**
     * Permet la suppression d'une propriété via l'objet comme si les attribut était public :
     *    unset($model->attribut);
     * @param string $offset
     */
    public function __unset($offset)
    {
        if (!$this->isReadonlyProperty($offset)) {
            $this->propertyUnset($offset);
        }
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetexists.php
     * @param string $offset
     * @return bool
     */
    public function offsetExists($offset)
    {
        return $this->propertyExists($offset);
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetget.php
     * @param string $offset
     * @return mixed
     */
    public function offsetGet($offset)
    {
        return $this->propertyGet($offset);
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetset.php
     * @param string $offset
     * @param mixed $value
     */
    public function offsetSet($offset, $value)
    {
        if (!$this->isReadonlyProperty($offset)) {
            $this->propertySet($offset, $value);
        }
    }

    /**
     * @see http://www.php.net/manual/fr/arrayaccess.offsetunset.php
     * @param string $offset
     */
    public function offsetUnset($offset)
    {
        if (!$this->isReadonlyProperty($offset)) {
            $this->propertyUnset($offset);
        }
    }


    /**
     * @see http://php.net/manual/en/iteratoraggregate.getiterator.php
     * @return \Traversable
     */
    public function getIterator()
    {
        return new \ArrayIterator($this->getArrayCopy());
    }

    /**
     * @return array
     */
    public function getModifiedProperties() : array
    {
        return $this->modifiedOffsets;
    }
}
