<?php


namespace POM\PredefinedType;

/**
 * Class JsonArray
 * @package POM\PredefinedType
 */
class JsonArray implements PredefinedTypeInterface
{

    /** @var  array */
    private $dataset;

    /**
     * JsonArray constructor.
     * @param array|string|null $dataset
     * @throws \InvalidArgumentException
     * @throws \DomainException
     */
    public function __construct($dataset = null)
    {
        if (is_string($dataset)) {
            $this->sqlUnserialize($dataset);
        } elseif (is_array($dataset)) {
            $this->dataset = array_values($dataset ?? []);
            array_walk($this->dataset, [$this, 'validateItem']);
        } else {
            $this->dataset = [];
        }
    }

    /**
     * @return array
     */
    public function getArrayCopy() : array
    {
        return $this->dataset;
    }

    /**
     * @param mixed $value
     * @throws \InvalidArgumentException
     */
    private function validateItem($value)
    {
        if (!is_scalar($value) || empty($value)) {
            throw new \InvalidArgumentException('the append value must be scalar & not empty');
        }
    }

    /**
     * @return string|null
     */
    public function sqlSerialize()
    {
        if (!empty($this->dataset)) {
            $values = array_map(function ($item) {
                $item = addcslashes($item, "'");
                return "'$item'";
            }, $this->dataset);
            return sprintf('Json_Array(%s)', implode(',', $values));
        }
        return null;
    }

    /**
     * @param string $serialized
     * @return void
     * @throws \DomainException
     */
    public function sqlUnserialize($serialized)
    {
        $data = \json_decode((string)$serialized, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \DomainException(sprintf(
                'Unable to decode JSON to data in %s: %s',
                __CLASS__,
                json_last_error_msg()
            ));
        }

        $this->dataset = (array)$data;
    }

    /**
     * @param mixed $value a scalar value
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function append($value)
    {
        $this->validateItem($value);
        if (empty($this->dataset)
            || !in_array($value, $this->dataset, true)
        ) {
            $this->dataset[] = $value;
        }
        return $this;
    }

    /**
     * @param callable|null $sortfunction
     * @return $this
     */
    public function sort(callable $sortfunction = null)
    {
        if ($sortfunction === null) {
            $sortfunction = 'sort';
        }
        $sortfunction($this->dataset);
        return $this;
    }

    /**
     * @param array $dataset
     * @return $this
     * @throws \InvalidArgumentException
     */
    public function exchangeArray(array $dataset)
    {
        $this->dataset = array_values($dataset);
        array_walk($this->dataset, [$this, 'validateItem']);
        return $this;
    }
}