<?php

namespace POM;

use POM\Service\AdapterPdoAbstract;

/**
 * Class MapperPdoAbstract
 * @package POM
 */
abstract class MapperPdoAbstract extends MapperSqlAbstract
{

    /**
     * MapperPdoAbstract constructor.
     * @param AdapterPdoAbstract $service
     * @param string $entityTable
     * @param array|string $entityPrimaries
     */
    public function __construct(AdapterPdoAbstract $service, $entityTable, $entityPrimaries)
    {
        parent::__construct($service, $entityTable, $entityPrimaries);
    }

    /**
     * @return \POM\Service\AdapterInterface|\POM\Service\AdapterPdoAbstract
     */
    public function getService()
    {
        return parent::getService();
    }

    /**
     * @inheritdoc
     */
    protected function quoteString($value)
    {
        return $this->getService()->quoteString($value);
    }


    /**
     * Sauve une liste de document et renvoi le nombre de ligne affecté
     * @param DomainObjectInterface $object
     * @param array $documents
     * @param bool $dryRun
     * @return int
     */
    public function bulkSave(
        DomainObjectInterface $object,
        array $documents,
        $dryRun = false
    ) {
        $bulk = [];
        foreach ($documents as $document) {
            $item = clone $object;
            $bulk[] = [
                $this->getSaveQuery($item->populate($document, true), $bindings),
                $bindings,
            ];
        }

        // execution du bulk de save
        return $this->getService()->execInTransaction($bulk, $lastInsertId, $dryRun);
    }

    /**
     * Insert une liste de document et renvoi le nombre de ligne affecté
     * @param DomainObjectInterface $object
     * @param array $documents
     * @param null|int $lastInsertId
     * @param bool $dryRun
     * @return int
     */
    public function bulkInsert(
        DomainObjectInterface $object,
        array $documents,
        &$lastInsertId = null,
        $dryRun = false
    ) {
        $bulk = [];
        foreach ($documents as $document) {
            $item = clone $object;
            $bulk[] = [
                $this->getInsertQuery($item->populate($document, true), $bindings),
                $bindings,
            ];
        }

        // execution du bulk d'insert
        return $this->getService()->execInTransaction($bulk, $lastInsertId, $dryRun);
    }

    /**
     * Met a jour une liste d'item
     * @param DomainObjectInterface[] $documents
     * @param bool $dryRun
     * @return int
     * @throws \UnexpectedValueException
     */
    public function bulkUpdate(
        array $documents,
        $dryRun = false
    ) {
        $bulk = [];
        foreach ($documents as $document) {
            if (!$document instanceof DomainObjectInterface) {
                throw new \UnexpectedValueException('objects must be a DomainObjectInterface list');
            }
            $bulk[] = [
                $this->getUpdateQuery($document, $bindings),
                $bindings,
            ];
        }

        // execution du bulk d'update
        return $this->getService()->execInTransaction($bulk, $lastInsertId, $dryRun);
    }
}
