<?php

namespace POM\Service;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;

/**
 * Class CursorPdo
 * @package POM\Service
 */
class CursorPdo implements \SeekableIterator, \Countable, LoggerAwareInterface
{

    use LoggerAwareTrait;

    /**
     * @var \PDOStatement
     */
    private $statement;

    /**
     * @var array
     */
    private $row = false;

    /**
     * @var int
     */
    private $position = -1;

    /**
     * @var int
     */
    private $total;

    /**
     * @var array
     */
    private $callback;

    /**
     * @var array
     */
    private $bindings;

    /**
     * @var AdapterPdoAbstract
     */
    private $pdoAdapter;

    /**
     * @var string
     */
    private $query;

    /**
     * @var bool
     */
    private $seekable;

    /**
     * @var int
     */
    private $limit;

    /**
     * @var int
     */
    private $offset = 0;

    /**
     * @var int
     */
    private $fetchMode;


    /**
     * @param AdapterPdoAbstract $pdoAdapter
     * @param string $query
     * @param array $bindings
     * @param int $fetchMode
     */
    public function __construct(
        AdapterPdoAbstract $pdoAdapter,
        $query,
        array $bindings = [],
        $fetchMode = \PDO::FETCH_ASSOC
    ) {
        // charge les données
        $this->fetchMode = $fetchMode;
        $this->bindings = $bindings;
        $this->pdoAdapter = $pdoAdapter;
        $this->query = $query;
        // check si on pe seek via la limit mysql ou si on seek a la main
        $this->seekable = false;
        if (!preg_match('@LIMIT +\d+, *\d+@i', $query)) {
            $this->seekable = true;
        }
        $this->logger = null;
    }

    /**
     * ferme le curseur
     */
    public function __destruct()
    {
        $this->closeCursor();
    }

    /**
     * @return void
     */
    public function closeCursor()
    {
        if ($this->statement) {
            $this->statement->closeCursor();
            $this->statement = null;
        }
    }

    /**
     * @return array
     */
    public function getArrayCopy() : array
    {
        return iterator_to_array($this, false);
    }

    /**
     * @param callable $callback
     * @param array $args
     */
    public function setRowHandler(callable $callback, array $args = [])
    {
        $this->callback = [$callback, $args];
    }


    /**
     * Return the current element
     * @link http://php.net/manual/en/iterator.current.php
     * @return mixed Can return any type.
     */
    public function current()
    {
        // si on a pas encore avancé...
        if (!$this->row) {
            $this->next();
        }
        $this->debug('ask for current at position ' . $this->position);
        // execution du handler de ligne
        if ($this->row !== false && $this->callback) {
            return \call_user_func_array(
                $this->callback[0],
                array_merge([$this->row], $this->callback[1])
            );
        }
        return $this->row;
    }

    /**
     * Charge la valeur du total
     * @return void
     */
    public function executeTotal()
    {
        if ($this->total === null) {
            // TODO : supprime de la requete la partie avec order by
            $query = $this->query;
            $query = 'SELECT COUNT(*) AS Total FROM (' . $query . ') AS t';
            $statement = $this->pdoAdapter->getStatementForQuery($query);
            $this->pdoAdapter->bindValue($statement, $this->bindings);
            $statement->execute();
            $statement->setFetchMode(\PDO::FETCH_COLUMN, 0);
            $this->total = $statement->fetch();
            $statement->closeCursor();
            $this->debug('execute total for query with total ' . $this->total);
        }
    }

    /**
     * Execute la requete réelement sur le serveur
     * @return void
     */
    public function execute()
    {
        $query = $this->query;
        if ($this->isSeekable()) {
            if (!$this->limit) {
                $this->executeTotal();
            }
            $query .= " LIMIT $this->offset," . ($this->limit ?: ($this->total - $this->position));
        }
        $this->statement = $this->pdoAdapter->getStatementForQuery($query);
        $this->pdoAdapter->bindValue($this->statement, $this->bindings);
        $this->statement->execute();
        $this->debug('execute query with total : ' . $this->statement->rowCount());
    }

    /**
     * Move forward to next element
     * @link http://php.net/manual/en/iterator.next.php
     * @return void Any returned value is ignored.
     */
    public function next()
    {
        if (!$this->statement) {
            $this->execute();
        }

        if (!$this->statement || $this->statement->rowCount() === 0) {
            $this->row = false;
        } else {
            $this->row = $this->statement->fetch(
                $this->fetchMode,
                \PDO::FETCH_ORI_NEXT
            );
            $this->position++;
        }
        $this->debug('next pdo cursor to position ' . $this->position);
    }

    /**
     * Return the key of the current element
     * @link http://php.net/manual/en/iterator.key.php
     * @return mixed scalar on success, or null on failure.
     */
    public function key()
    {
        return $this->position;
    }

    /**
     * Checks if current position is valid
     * @link http://php.net/manual/en/iterator.valid.php
     * @return boolean The return value will be casted to boolean and then evaluated.
     * Returns true on success or false on failure.
     */
    public function valid() : bool
    {
        if (!$this->statement) {
            $this->executeTotal();
        }
        $valid = $this->statement ? $this->row !== false : (
        $this->total ? $this->position < $this->total : false
        );
        $this->debug('check validity for position ' . $this->position . ' is ' . $valid);
        return $valid;
    }

    /**
     * Rewind the Iterator to the first element
     * @link http://php.net/manual/en/iterator.rewind.php
     * @throws \OutOfBoundsException
     * @return void Any returned value is ignored.
     */
    public function rewind()
    {
        $this->debug('rewind pdo cursor from position ' . $this->position);
        if ($this->position > 0) {
            $this->closeCursor();
            $this->position = -1;
        }
    }

    /**
     * @inheritdoc
     */
    public function count() : int
    {
        $this->executeTotal();
        return $this->total;
    }

    /**
     * Seeks to a position
     * @link http://php.net/manual/en/seekableiterator.seek.php
     * @param int $position The position to seek to.
     * @return void
     */
    public function seek($position)
    {
        $this->debug('seek pdo cursor to position ' . $position);
        if ($this->isSeekable()) {
            if ($this->statement && $this->offset !== (int)$position) {
                $this->closeCursor();
            }
            $this->offset = (int)$position;
            $this->position = (int)$position;
        } else {
            while ($this->position < $position) {
                $this->next();
            }
        }
    }

    /**
     * @param int $position
     * @return $this
     */
    public function offset($position)
    {
        $this->seek($position);
        return $this;
    }

    /**
     * @param int $byPage
     * @return $this
     */
    public function limit($byPage)
    {
        $this->limit = $byPage;
        return $this;
    }

    /**
     * @return \PDOStatement|null
     */
    public function getStatement()
    {
        return $this->statement;
    }

    /**
     * @return boolean
     */
    public function isSeekable() : bool
    {
        return $this->seekable;
    }

    /**
     * @param string $msg
     * @param array $opts
     */
    private function debug(string $msg, array $opts = [])
    {
        if ($this->logger instanceof LoggerInterface) {
            $this->logger->debug('[' . $this->query . '] ' . $msg, $opts + [
                    'row_type' => gettype($this->row),
                ]);
        }
    }
}
