<?php

namespace POM\Service;

use POM\IdentityMap;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\NullLogger;

/**
 * Class AdapterPdoAbstract
 * @package POM\Service
 */
abstract class AdapterPdoAbstract extends AdapterAbstract implements LoggerAwareInterface
{

    use LoggerAwareTrait;

    /**
     * @var int
     */
    private $nestLevel = 0;

    /**
     * @var bool
     */
    protected $nestableTransaction = false;

    /**
     * @var array
     */
    protected $dbAccess;

    /**
     * @var \PDO
     */
    protected $dbHandler;

    /**
     * @var IdentityMap
     */
    protected $stmtMap;

    /**
     * @param string $dsn
     * @param string $user
     * @param string $pass
     * @param array $opts
     */
    public function __construct($dsn, $user, $pass, array $opts = [])
    {
        $this->dbAccess['dsn'] = $dsn;
        $this->dbAccess['user'] = $user;
        $this->dbAccess['pass'] = $pass;
        $this->dbAccess['opts'] = [
                \PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION,
            ] + $opts;
        // on utilise une identity map pour stocké les statements afin de les réutiliser
        $this->stmtMap = new IdentityMap();
        $this->logger = new NullLogger();
        // close handler pour fermeture de la connection
        register_shutdown_function([$this, 'deconnect']);
    }

    /**
     * detruit la connection actuellement ouverte
     */
    public function __destruct()
    {
        $this->deconnect();
    }

    /**
     * Déconnection de la DB
     */
    public function deconnect()
    {
        if ($this->dbHandler instanceof \PDO) {
            $this->logger->debug('PDOAdapter DECONNECT', [
                'access_infos' => $this->dbAccess,
            ]);
            $this->dbHandler = null;
        }
    }

    /**
     * @return bool
     */
    public function beginTransaction() : bool
    {
        if (!$this->nestableTransaction || $this->nestLevel++ === 0) {
            $result = $this->getDbHandler()->beginTransaction();
        } else {
            $this->exec("SAVEPOINT LEVEL{$this->nestLevel}");
            $result = true;
        }
        $this->logger->info("PDOAdapter BEGIN Transaction >> $this->nestLevel TO UP");
        return $result;
    }

    /**
     * @return bool
     */
    public function commitTransaction() : bool
    {
        if (!$this->nestableTransaction || $this->nestLevel <= 1) {
            $result = $this->getDbHandler()->commit();
        } else {
            $this->exec("RELEASE SAVEPOINT LEVEL{$this->nestLevel}");
            $result = true;
        }
        $this->nestLevel--;
        $this->logger->info("PDOAdapter COMMIT Transaction >> $this->nestLevel TO DOWN");
        return $result;
    }

    /**
     * @return bool
     */
    public function rollBackTransaction() : bool
    {
        if (!$this->nestableTransaction || $this->nestLevel <= 1) {
            $result = $this->getDbHandler()->rollBack();
        } else {
            $this->exec("ROLLBACK TO SAVEPOINT LEVEL{$this->nestLevel}");
            $result = true;
        }
        $this->nestLevel--;
        $this->logger->info("PDOAdapter ROLLBACK Transaction >> $this->nestLevel TO DOWN");
        return $result;
    }

    /**
     * Connection de la DB, si erreur Exception est envoyé, sinon true si connexion requise,
     * false si déja effectué
     * @return bool
     */
    public function connect() : bool
    {
        if (!$this->dbHandler instanceof \PDO) {
            $this->dbHandler = new \PDO(
                $this->dbAccess['dsn'],
                $this->dbAccess['user'],
                $this->dbAccess['pass'],
                $this->dbAccess['opts']
            );
            $this->logger->debug('PDOAdapter CONNECT TO Database', [
                'access_infos' => $this->dbAccess,
            ]);
            return true;
        }
        return false;
    }

    /**
     * @param string $string
     * @return string
     * @throws \InvalidArgumentException
     */
    public function quoteString($string) : string
    {
        if (!\is_string($string)) {
            throw new \InvalidArgumentException('this function can only quote string.');
        }
        return $this->getDbHandler()->quote($string);
    }

    /**
     * Renvoi l'instance de DBHandler
     * @return \PDO
     */
    public function getDbHandler()
    {
        $this->connect();
        return $this->dbHandler;
    }

    /**
     * Renvoi le statement associé a une query
     * @param string $query
     * @return \PDOStatement
     */
    public function getStatementForQuery($query)
    {
        // on verifie que le statement n'est pas deja présent
        if (!$this->stmtMap->hasId($query)) {
            $stmt = $this->getDbHandler()->prepare($query, [
                \PDO::ATTR_CURSOR => \PDO::CURSOR_SCROLL,
            ]);
            $this->stmtMap->storeObject($query, $stmt);
        } else {
            $stmt = $this->stmtMap->getObject($query);
        }
        return $stmt;
    }

    /**
     * Assign a un statement les bind
     * @param \PDOStatement $stmt
     * @param array $bind
     * @return void
     */
    public function bindValue(\PDOStatement $stmt, array $bind)
    {
        $bind = array_map(function ($value, $key) {
            $sValueType = \PDO::PARAM_STR;
            if (\is_int($value)) {
                $value = (int)$value;
            } elseif ($value instanceof \DateTime) {
                $value = $value->format('Y-m-d H:i:s');
            }
            if (\is_bool($value)) {
                $sValueType = \PDO::PARAM_BOOL;
                $value = $value ? '1' : '0';
            } elseif (null === $value) {
                $sValueType = \PDO::PARAM_NULL;
            } elseif (\is_int($value)) {
                $sValueType = \PDO::PARAM_INT;
            }
            return [is_numeric($key) ? $key + 1 : $key, (string)$value, $sValueType];
        }, $bind, array_keys($bind));

        // lecture du binding
        foreach ($bind as list($k, $v, $type)) {
            $stmt->bindValue($k, $v, $type);
        }
    }


    /**
     * @inheritdoc
     * @return CursorPdo
     */
    public function fetch($query, array $bind = []) : CursorPdo
    {
        $cursor = new CursorPdo($this, $query, $bind);
        return $cursor;
    }

    /**
     * @inheritdoc
     */
    public function fetchOne($query, array $bind = [])
    {
        $statement = $this->getStatementForQuery($query);
        $this->bindValue($statement, $bind);
        $statement->execute();
        $result = $statement->fetch(\PDO::FETCH_ASSOC);
        return $result;
    }

    /**
     * @inheritdoc
     * @return CursorPdo
     */
    public function fetchColumn($column, $query, array $bind = []) : CursorPdo
    {
        $cursor = new CursorPdo($this, $query, $bind, \PDO::FETCH_COLUMN);
        $cursor->execute();
        if (null !== ($stmt = $cursor->getStatement())) {
            $stmt->setFetchMode(\PDO::FETCH_COLUMN, $column);
        }
        return $cursor;
    }

    /**
     * Execute une liste de requete dans une transaction et renvoi le nombre total de ligne affecté
     * si $lastInsertId est fourni il est rempli avec le dernier ID inséré
     * @param array $queryList [ ['SQL QUERY', ['PARAM'=>'VALUE', ...]], 'SQL QUERY', ... ]
     * @param string $lastInsertId
     * @param bool $dryRun
     * @return int
     * @throws \PDOException
     */
    public function execInTransaction(array $queryList, &$lastInsertId = null, $dryRun = false) : int
    {
        $rowCount = 0;
        $this->beginTransaction();
        foreach ($queryList as $queryParam) {
            if (\is_string($queryParam)) {
                $rowCount += $this->exec($queryParam, []);
            } elseif (\is_array($queryParam) && count($queryParam) >= 2) {
                $rowCount += $this->exec($queryParam[0], $queryParam[1]);
            }
        }
        if ($dryRun !== true) {
            $this->commitTransaction();
        } else {
            $this->rollBackTransaction();
        }
        $lastInsertId = $this->getDbHandler()->lastInsertId();
        return $rowCount;
    }

    /**
     * Effectue une requete et renvoi le nobre de ligne affecté,
     * si $lastInsertId est fourni il est rempli avec le dernier ID inséré
     * @param string $query
     * @param array $bind
     * @param string $lastInsertId
     * @return int
     */
    public function exec($query, array $bind = [], &$lastInsertId = null) : int
    {
        $this->logger->debug('PDOAdapter EXEC Query : ' . $query, [
            'bindings' => $bind,
        ]);
        $stmt = $this->getStatementForQuery($query);
        !$bind ?: $this->bindValue($stmt, $bind);
        if ($result = $stmt->execute()) {
            $lastInsertId = $this->getDbHandler()->lastInsertId();
            return $stmt->rowCount();
        }
        return 0;
    }
}
