<?php

namespace POM;

use POM\PredefinedType\PredefinedTypeInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;

/**
 * Class MapperSqlAbstract
 * @package POM
 */
abstract class MapperSqlAbstract extends MapperAbstract implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    /**
     * @param string $value
     * @return string
     */
    abstract protected function quoteString($value);

    /**
     * @param string $type
     * @return callable|null
     */
    protected function getFilterCallback(string $type): ?callable
    {
        return null;
    }

    /**
     * Returns the SQL condition to identify this model
     * @param array $entities
     * @return array (SQL_WHERE, BINDINGS, IDMAP)
     */
    final protected function getEntityCondition(array $entities): array
    {
        $primaries = $this->getEntityPrimaries();
        $entitiesFiltered = array_intersect_key($entities, array_flip($primaries));
        $condition = $this->getCondition($entitiesFiltered);
        $condition[] = implode('-', $entitiesFiltered);
        return $condition;
    }

    /**
     * @param mixed $value
     * @return string|array
     */
    final protected function convertEntityValue($value)
    {
        // si serializable
        if ($value instanceof SerializableSqlInterface) {
            $value = $value->sqlSerialize();
        } elseif ($value instanceof \JsonSerializable) {
            $value = json_encode($value);
        } elseif ($value instanceof \Serializable) {
            $value = $value->serialize();
        }
        return $value;
    }

    /**
     * @param string $key
     * @return string
     */
    protected function convertKeyForBindings($key): string
    {
        return ":$key";
    }

    /**
     * @param array $entities
     * @return array  (SQL_WHERE, BINDINGS)
     */
    protected function getUpdateSet(array $entities): array
    {
        return $this->getCondition($entities, ' ,', false);
    }

    /**
     * Returns the SQL condition
     * @param array $entities
     * @param string $glue default ' AND '
     * @param bool $authorizeLike
     * @return array (SQL_WHERE, BINDINGS)
     */
    final protected function getCondition(array $entities, $glue = ' AND ', bool $authorizeLike = true): array
    {
        // generation des valeur modifié
        $values = array_map([$this, 'convertEntityValue'], $entities);
        $keyEntities = array_keys($entities);

        // generation des placeholders
        $placeholders = array_combine(
            $keyEntities,
            array_map(
                static function ($key, $value) {
                    $key = strtolower(str_replace('.', '_', $key));
                    if (is_array($value)) {
                        $keys = [];
                        for ($i = 1, $iCount = count($value); $i <= $iCount; $i++) {
                            $keys[] = $key . '_' . $i;
                        }
                        return $keys;
                    }
                    return $key;
                },
                $keyEntities,
                $values
            )
        );

        // gestion de la chaine de conditions
        $conditions = array_map(
            function ($key, $placeholder, $value) use ($authorizeLike) {
                if (false === strpos($key, '`.`')) {
                    $key = str_replace('.', '`.`', $key);
                }
                if (is_array($value)) {
                    return !empty($value) ? sprintf(
                        '`%s` IN (' . implode(
                            ', ',
                            array_map([$this, 'convertKeyForBindings'], $placeholder)
                        ) . ')',
                        $key
                    ) : null;
                }

                $operator = is_scalar($value)
                && $authorizeLike === true
                && (preg_match('@^%@', $value) === 1 || preg_match('@%$@', $value) === 1)
                    ? 'LIKE'
                    : '=';
                $placeholder = $this->convertKeyForBindings($placeholder);
                return sprintf("`%s` $operator $placeholder", (string)$key);
            },
            $keyEntities,
            $placeholders,
            $values
        );

        // generation du ntableau de binding
        $bindings = [];
        foreach ($placeholders as $phName => $placeholder) {
            if (is_array($placeholder)) {
                foreach ($placeholder as $kph => $ph) {
                    $bindings[$ph] = $values[$phName][$kph];
                }
            } else {
                $bindings[$placeholder] = $values[$phName];
            }
        }
        $sql = implode($glue, array_filter($conditions));
        ksort($bindings);
        return [$sql, $bindings];
    }


    /**
     * Charge les données dans $object de l'element trouvé via son id,
     * renvoi true/false selon la réussite de la requete
     * @param mixed $id
     * @param DomainObjectInterface $object
     * @return bool
     */
    public function fetchById($id, DomainObjectInterface &$object)
    {
        if (!is_array($id)) {
            $id = array_combine($this->getEntityPrimaries(), [$id]);
        }
        [$where, $bindings] = $this->getEntityCondition($id);
        $query = 'SELECT * FROM `' . $this->getEntityTable() . '` WHERE ' . $where;
        // on ne sauvegarde les données que si elle sont rempli
        if (!($data = $this->service->fetchOne($query, $bindings))) {
            return false;
        }
        $this->populate($object, $data, true);
        return true;
    }

    /**
     * Supprime un element de la DB via son ID,
     * renvoi true/false selon la réussite de la requete
     * @param mixed $id
     * @return bool
     */
    public function removeById($id)
    {
        if (!is_array($id)) {
            $id = array_combine($this->getEntityPrimaries(), [$id]);
        }
        [$where, $bindings] = $this->getEntityCondition($id);
        $query = 'DELETE FROM `' . $this->getEntityTable() . '` WHERE ' . $where;
        $this->service->exec($query, $bindings);
        return true;
    }

    /**
     * @param array $entities
     * @return array $fields, $binderKeys, $bindings
     */
    protected function convertEntitiesForBinding(array $entities): array
    {
        $fields = $binderKeys = $bindings = [];
        foreach ($entities as $field => $entity) {
            $value = $this->convertEntityValue($entity);
            $bindKey = $this->convertKeyForBindings($field);
            if ($entity instanceof PredefinedTypeInterface) {
                $value = $entity->prepareForRecord();
                if ($value !== null) {
                    $binderKeys[] = $value;
                    $fields[] = '`' . str_replace('.', '`.`', $field) . '`';
                }
            } else {
                $binderKeys[] = $bindKey;
                $bindings[$bindKey] = $value;
                $fields[] = '`' . str_replace('.', '`.`', $field) . '`';
            }
        }
        return [$fields, $binderKeys, $bindings];
    }

    /**
     * @param DomainObjectInterface $object
     * @param array $bindings
     * @param callable|null $filterCallback
     * @return null|string
     */
    final protected function getSaveQuery(
        DomainObjectInterface $object,
        &$bindings = [],
        callable $filterCallback = null
    ) {
        if ($filterCallback) {
            $entities = array_filter($object->getArrayCopy(), $filterCallback);
        } else {
            $entities = array_filter($object->getArrayCopy());
        }
        [$fields, $binderKeys, $bindings] = $this->convertEntitiesForBinding($entities);
        if (!empty($bindings)) {
            $query = sprintf(
                'REPLACE INTO `%s` (%s) VALUES (%s)',
                $this->getEntityTable(),
                implode(', ', $fields),
                implode(', ', $binderKeys)
            );
            return $query;
        }
        return null;
    }

    /**
     * Sauvegarde l'objet dans la DB, l'$objet est mis a jour selon les modif appliqué par la DB
     * (insert: id...) renvoi true/false selon la réussite de la requete
     * @param DomainObjectInterface $object
     * @return bool
     */
    public function save(DomainObjectInterface &$object)
    {
        $query = $this->getSaveQuery($object, $values, $this->getFilterCallback('replace'));
        if ($query !== null) {
            $this->service->exec($query, $values);
            return true;
        }
        return $this->insert($object);
    }

    /**
     * @param DomainObjectInterface $object
     * @param array $bindings
     * @param bool $ignore
     * @param callable|null $filterCallback
     * @return null|string
     */
    final protected function getInsertQuery(
        DomainObjectInterface $object,
        &$bindings = [],
        $ignore = false,
        callable $filterCallback = null
    ) {
        if ($filterCallback !== null) {
            $entities = array_filter($object->getArrayCopy(), $filterCallback);
        } else {
            $entities = array_filter($object->getArrayCopy());
        }
        $keys = array_keys($entities);
        $bindings = array_map([$this, 'convertEntityValue'], $entities);
        if (!empty($bindings)) {
            $query = sprintf(
                'INSERT %sINTO `%s` (`%s`) VALUES (%s)',
                $ignore ? 'IGNORE ' : '',
                $this->getEntityTable(),
                implode('`, `', $keys),
                implode(', ', array_map([$this, 'convertKeyForBindings'], $keys))
            );
            return $query;
        }
        return null;
    }

    /**
     * Insert l'objet dans la DB, l'$objet est mis a jour selon les modif appliqué par la DB
     * (insert: id...) si l'objet existe déjà on renvoi un exception renvoi true/false selon la
     * réussite de la requete
     * @param DomainObjectInterface $object
     * @return bool
     */
    public function insert(DomainObjectInterface &$object)
    {
        $query = $this->getInsertQuery($object, $values, $this->getFilterCallback('insert'));

        /*error_log("[SQL] insert in table " . $this->getEntityTable() . " with query : "
            . $query . " and bindings " . json_encode($bindings));*/

        if ($query !== null && $this->service->exec($query, $values, $insertId)) {
            // insert de la clé
            $object[$this->getEntityPrimaries()[0]] = $insertId;
            return true;
        }
        return false;
    }

    /**
     * @param DomainObjectInterface $object
     * @param array $bindings
     * @return null|string
     */
    final protected function getUpdateQuery(DomainObjectInterface $object, &$bindings = [])
    {
        [$condition, $bindings] = $this->getEntityCondition($object->getArrayCopy());

        if (!empty($condition) && !empty($bindings)) {
            $primaries = $this->getEntityPrimaries();
            $entities = array_diff_key($object->getArrayCopy(true), array_flip($primaries));
            if (empty($entities)) {
                return null;
            }
            [$update, $updateBindings] = $this->getUpdateSet($entities);
            if (empty($update) || empty($updateBindings)) {
                return null;
            }
            $bindings = array_merge($updateBindings, $bindings);

            $query = sprintf(
                'UPDATE `%s` SET %s WHERE %s',
                $this->getEntityTable(),
                $update,
                $condition
            );

            return $query;
        }

        return null;
    }

    /**
     * Update l'objet dans la DB, l'$objet est mis a jour selon les modif appliqué par la DB
     * (insert: id...) si l'objet n'existe pas déjà on renvoi un exception renvoi true/false selon
     * la réussite de la requete
     * @param DomainObjectInterface $object
     * @return mixed
     */
    public function update(DomainObjectInterface &$object)
    {
        $query = $this->getUpdateQuery($object, $bindings);
        if ($query !== null) {
            /*error_log("[SQL] update in table " . $this->getEntityTable() . " with query : "
                . $query . " and bindings " . json_encode($bindings));*/
            $result = $this->service->exec($query, $bindings);
            if ($result > 1) {
                trigger_error(
                    "too many element has been update : $result with request $query ("
                    . json_encode($bindings) . ')',
                    E_USER_WARNING
                );
            } elseif ($result === 0) {
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->debug(
                        "nothing to update with request $query",
                        [
                            'bindings' => $bindings,
                            'object' => $object->getArrayCopy(),
                        ]
                    );
                }
            }
        } else {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->debug(
                    'nothing to update on table ' . $this->getEntityTable(),
                    [
                        'object' => $object->getArrayCopy(),
                    ]
                );
            }
        }
        return true;
    }

    /**
     * Supprime l'objet de la DB, si l'objet n'existe pas déjà on renvoi un exception
     * renvoi true/false selon la réussite de la requete
     * @param DomainObjectInterface $object
     * @return mixed
     */
    public function remove(DomainObjectInterface $object)
    {
        $id = [];
        foreach ($this->getEntityPrimaries() as $primaryKey) {
            $id[$primaryKey] = $object[$primaryKey];
        }
        return empty($id) ? false : $this->removeById($id);
    }
}
